﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_FileSystemForDebug.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>

namespace nn { namespace fs { namespace fsa {

    class IFile;
    class IDirectory;

    enum class QueryId
    {
        SetConcatenationFileAttribute,
    };

    /**
    * @brief    各オブジェクトのファクトリおよびファイルシステム操作のためのインターフェースです。
    *
    * @details  全ての関数同士の呼び出しについて、複数のスレッドから同時に呼び出された場合に正しく動作するよう実装する必要があります。
    */
    class IFileSystem
    {
    public:

        Result CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(size >= 0,       nn::fs::ResultOutOfRange());  // TODO: ResultInvalidSize?
            return DoCreateFile(path, size, option);
        }

        Result CreateFile(const char* path, int64_t size) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(size >= 0,       nn::fs::ResultOutOfRange());  // TODO: ResultInvalidSize?
            return CreateFile(path, size, 0);
        }

        Result DeleteFile(const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            return DoDeleteFile(path);
        }

        Result CreateDirectory(const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            return DoCreateDirectory(path);
        }

        Result DeleteDirectory(const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            return DoDeleteDirectory(path);
        }

        Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            return DoDeleteDirectoryRecursively(path);
        }

        Result CleanDirectoryRecursively(const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            return DoCleanDirectoryRecursively(path);
        }

        Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(currentPath != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(newPath     != nullptr, nn::fs::ResultInvalidPath());
            return DoRenameFile(currentPath, newPath);
        }

        Result RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(currentPath != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(newPath     != nullptr, nn::fs::ResultInvalidPath());
            return DoRenameDirectory(currentPath, newPath);
        }

        Result GetEntryType(DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path     != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());
            return DoGetEntryType(outValue, path);
        }

        Result OpenFile(std::unique_ptr<IFile>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path     != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());
            NN_FSP_REQUIRES(
                (mode & (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)) != 0
                && (mode & ~(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)) == 0,
                ResultInvalidArgument(),
                "OpenMode_Read or OpenMode_Write is required."
            );
            return DoOpenFile(outValue, path, mode);
        }

        Result OpenDirectory(std::unique_ptr<IDirectory>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path     != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());
            NN_FSP_REQUIRES(
                (mode & nn::fs::OpenDirectoryMode_All) != 0
                && (mode & ~(nn::fs::OpenDirectoryMode_All | nn::fs::OpenDirectoryModePrivate_NotRequireFileSize)) == 0,
                ResultInvalidArgument(),
                "OpenDirectoryMode_Directory, OpenDirectoryMode_File or OpenDirectoryMode_All is required."
            );
            return DoOpenDirectory(outValue, path, mode);
        }

        Result Commit() NN_NOEXCEPT
        {
            return DoCommit();
        }

        Result CommitProvisionally(int64_t counter) NN_NOEXCEPT
        {
            return DoCommitProvisionally(counter);
        }

        Result Rollback() NN_NOEXCEPT
        {
            return DoRollback();
        }

        Result GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path     != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());
            return DoGetFreeSpaceSize(outValue, path);
        }

        Result GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path     != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());
            return DoGetTotalSpaceSize(outValue, path);
        }

        Result Flush() NN_NOEXCEPT
        {
            return DoFlush();
        }

        Result GetFileTimeStampRaw(FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(outTimeStamp != nullptr, nn::fs::ResultNullptrArgument());
            return DoGetFileTimeStampRaw(outTimeStamp, path);
        }

        Result QueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, QueryId queryId, const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultInvalidPath());
            return DoQueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, path);
        }


    public:
        /**
        * @brief    nn::fs::fsa::Unregister に応じて呼び出されるデストラクタです。
        *
        * @details  実装クラスはファイルシステムの終了処理をデストラクタで実装してください。
                    nn::fs::fsa::Register を呼び出すと IFileSystem のインスタンスの所有権が FS ライブラリに委譲されます。
                    FS ライブラリがインスタンスを破棄する際に delete が呼び出されます。
        */
        virtual ~IFileSystem() NN_NOEXCEPT {}

    private:
        /**
        * @brief    nn::fs::CreateFile に応じて呼び出されるファイル作成関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルの作成処理を実装してください。
        *           path 引数には、nn::fs::CreateFile 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::CreateFile 関数から委譲されます。
        */
        virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::DeleteFile に応じて呼び出されるファイル削除関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルの削除処理を実装してください。
        *           path 引数には、nn::fs::DeleteFile 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::CreateDirectory に応じて呼び出されるディレクトリ作成関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリの作成処理を実装してください。
        *           path 引数には、nn::fs::CreateDirectory 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::DeleteDirectory に応じて呼び出されるディレクトリ削除関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリの削除処理を実装してください。
        *           path 引数には、nn::fs::DeleteDirectory 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::DeleteDirectoryRecursively に応じて呼び出されるディレクトリ再帰削除関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリの再帰削除処理を実装してください。
        *           path 引数には、nn::fs::DeleteDirectoryRecursively 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::CleanDirectoryRecursively に応じて呼び出される子エントリ削除関数です。
        *
        * @details  実装クラスはこの関数の実装で子エントリの削除処理を実装してください。
        *           path 引数には、nn::fs::CleanDirectoryRecursively 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::RenameFile に応じて呼び出されるファイルリネーム関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルのリネーム処理を実装してください。
        *           currrentPath および newPath 引数には、nn::fs::RenameFile 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::RenameFile に応じて呼び出されるディレクトリリネーム関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリのリネーム処理を実装してください。
        *           currrentPath および newPath 引数には、nn::fs::RenameFile 関数に渡したパスの実体パスが渡されます。
        */
        virtual Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::GetEntryType に応じて呼び出されるファイルの存在確認関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルの存在確認処理を実装してください。
        *           path 引数には、nn::fs::GetEntryType 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::GetEntryType 関数から委譲されます。
        */
        virtual Result DoGetEntryType(DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::GetFreeSpaceSize に応じて呼び出される空き容量を取得する関数です。
        *
        * @details  実装クラスはこの関数の実装で空き容量取得処理を実装してください。
        *           path 引数には、nn::fs::GetFreeSpaceSize 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::GetFreeSpaceSize 関数から委譲されます。
        */
        virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
        {
            NN_UNUSED(outValue);
            NN_UNUSED(path);
            NN_RESULT_THROW(ResultNotImplemented());
        }

        /**
        * @brief    nn::fs::GetTotalSpaceSize に応じて呼び出される全体容量を取得する関数です。
        *
        * @details  実装クラスはこの関数の実装で全体容量取得処理を実装してください。
        *           path 引数には、nn::fs::GetTotalSpaceSize 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::GetTotalSpaceSize 関数から委譲されます。
        */
        virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
        {
            NN_UNUSED(outValue);
            NN_UNUSED(path);
            NN_RESULT_THROW(ResultNotImplemented());
        }

        /**
        * @brief    nn::fs::OpenFile に応じて呼び出されるファイルを開く関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルを開いて、そのファイルを操作するための
        *           IFile オブジェクトを生成してください。
        *           outValue 引数には、std::unique_ptr::reset 関数を呼び出して、生成した IFile
        *           オブジェクトをスマートポインタに設定してください。IFile オブジェクトの所有権は
        *           呼び出し側に委譲されます。
        *           path 引数には、nn::fs::OpenFile 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::OpenFile 関数から委譲されます。
        */
        virtual Result DoOpenFile(std::unique_ptr<IFile>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::OpenDirectory に応じて呼び出されるディレクトリを開く関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリを開いて、そのディレクトリを操作するための
        *           IDirectory オブジェクトを生成してください。
        *           outValue 引数には、std::unique_ptr::reset 関数を呼び出して、生成した IDirectory
        *           オブジェクトをスマートポインタに設定してください。IDirectory オブジェクトの所有権は
        *           呼び出し側に委譲されます。
        *           path 引数には、nn::fs::OpenDirectory 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::OpenDirectory 関数から委譲されます。
        */
        virtual Result DoOpenDirectory(std::unique_ptr<IDirectory>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::Commit に応じて呼び出されるファイルシステムのコミット関数です。
        *
        * @details  実装クラスはこの関数の実装でアトミックなコミット処理を実装してください。
        */
        virtual Result DoCommit() NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::CommitProvisionally に応じて呼び出されるファイルシステムの仮コミット関数です。
        *
        * @details  実装クラスはこの関数の実装でアトミックな仮コミット処理を実装してください。
        */
        virtual Result DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
        {
            NN_UNUSED(counter);
            NN_RESULT_THROW(ResultNotImplemented());
        }

        /**
        * @brief    nn::fs::Rollback に応じて呼び出されるファイルシステムのロールバック関数です。
        *
        * @details  実装クラスはこの関数の実装でアトミックなロールバック処理を実装してください。
        */
        virtual Result DoRollback() NN_NOEXCEPT
        {
            NN_RESULT_THROW(ResultNotImplemented());
        }

        /**
        * @brief    ファイルシステムのフラッシュ関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルシステムのフラッシュを実装してください。
        */
        virtual Result DoFlush()
        {
            NN_RESULT_THROW(ResultNotImplemented());
        }

        /**
        * @brief    nn::fs::GetTimeStampForDebug に応じて呼び出されるファイルのタイムスタンプ取得関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルのタイムスタンプ取得処理を実装してください。
        *           path 引数には、nn::fs::GetTimeStampForDebug 関数に渡したパスの実体パスが渡されます。
        *           その他の引数は nn::fs::GetTimeStampForDebug 関数から委譲されます。
        */
        virtual Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
        {
            NN_UNUSED(outTimeStamp);
            NN_UNUSED(path);
            NN_RESULT_THROW(ResultNotImplemented());
        }

        /**
        * @brief    エントリを対象とした特殊処理関数に応じて呼び出されるファイルシステムのクエリ関数です。
        *
        * @details  実装クラスはこの関数の実装で QueryId に応じた処理を実装してください。
        */
        virtual Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, QueryId queryId, const char* path) NN_NOEXCEPT
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_UNUSED(queryId);
            NN_UNUSED(path);
            NN_RESULT_THROW(ResultNotImplemented());
        }

    };

}}}
