﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>

namespace nn { namespace fs {
    class IEventNotifier;
}}

namespace nn { namespace fs {


/**
*   @brief      SD カードのルートに作成する Nintendo ルートディレクトリの名前です。
*/
const char* const SdCardNintendoRootDirectoryName = "Nintendo";

/**
*   @brief      SD カードを扱うファイルシステムを指定したマウント名でマウントします。
*
*   @param[in]  name       マウント名
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*/
Result MountSdCard(const char* name) NN_NOEXCEPT;

/**
*   @brief      SD カードの Speed mode の種類を示す列挙型です
*/
enum SdCardSpeedMode {
    SdCardSpeedMode_Identification, //!< Identification
    SdCardSpeedMode_DefaultSpeed,   //!< Default Speed
    SdCardSpeedMode_HighSpeed,      //!< High Speed
    SdCardSpeedMode_Sdr12,          //!< SDR12
    SdCardSpeedMode_Sdr25,          //!< SDR25
    SdCardSpeedMode_Sdr50,          //!< SDR50
    SdCardSpeedMode_Sdr104,         //!< SDR104
    SdCardSpeedMode_Ddr50,          //!< DDR50
    SdCardSpeedMode_Unknown         //!< 不明な Speed Mode です。
};

/**
 * @brief   SD カードの CID のサイズを表す定数です。
 */
const size_t SdCardCidSize = 16;

/**
*   @brief      SD カードの挿抜通知設定のためのインターフェイスを取得します。
*
*   @param[out] outValue   取得されたインターフェイス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result OpenSdCardDetectionEventNotifier(std::unique_ptr<IEventNotifier>* outValue) NN_NOEXCEPT;

/**
*   @brief      SD カードの挿抜状態を取得します。
*
*   @return     挿入されている場合は true、抜かれている場合は false が返ります。
*/
bool IsSdCardInserted() NN_NOEXCEPT;

/**
*   @brief      SD カードの Speed mode を取得します。
*
*   @param[out] outValue   取得した Speed mode の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - outValue が有効なメモリを指している。
*       - MountSdCardForDebug() により SD カードがマウント済みである。
*/
Result GetSdCardSpeedMode(SdCardSpeedMode* pOutValue) NN_NOEXCEPT;

/**
*   @brief      SD カードの CID を取得します。
*
*   @param[out] pOutBuffer  取得した CID の格納先
*   @param[in]  size        取得した CID を格納する領域のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre        size == SdmmcCidSize
*/
Result GetSdCardCid(void* outBuffer, size_t size) NN_NOEXCEPT;

/**
*   @brief      SD カードの CID から MID を取得します。
*
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     MID の値を返します。
*/
inline uint8_t GetMidFromSdCardCid(const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    return pSdCardCid[14];
}

/**
 * @brief   SD カードの CID の OID 文字列のバイトサイズを表す定数です。終端文字は含まれません。
 */
const size_t SdCardCidOidSize = 2;

/**
*   @brief      SD カードの CID から OID 文字列を取得します。終端文字は含まれません。
*
*   @param[out] pOutString  OID 文字列の格納先
*   @param[in]  outStringSize   OID 文字列を格納する領域のサイズ
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     なし
*/
inline void GetOidFromSdCardCid(char* pOutString, size_t outStringSize, const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    NN_ABORT_UNLESS(outStringSize >= SdCardCidOidSize);
    pOutString[1] = static_cast<char>(pSdCardCid[12]);
    pOutString[0] = static_cast<char>(pSdCardCid[13]);
}

/**
 * @brief   SD カードの CID の PNM 文字列のバイトサイズを表す定数です。終端文字は含まれません。
 */
const size_t SdCardCidPnmSize = 5;

/**
*   @brief      SD カードの CID から PNM 文字列を取得します。終端文字は含まれません。
*
*   @param[out] pOutString  PNM 文字列の格納先
*   @param[in]  outStringSize   PNM 文字列を格納する領域のサイズ
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     なし
*/
inline void GetPnmFromSdCardCid(char* pOutString, size_t outStringSize, const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    NN_ABORT_UNLESS(outStringSize >= SdCardCidPnmSize);
    const int SdCardCidPnmIndex = 7;
    for (size_t i = 0; i < SdCardCidPnmSize; i++)
    {
        pOutString[SdCardCidPnmSize - i - 1] =  static_cast<char>(pSdCardCid[SdCardCidPnmIndex + i]);
    }
}

/**
*   @brief      SD カードの CID から PRV を取得します。
*
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     PRV の値を返します。
*/
inline uint8_t GetPrvFromSdCardCid(const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    return pSdCardCid[6];
}

/**
*   @brief      SD カードの CID から PSN を取得します。
*
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     PSN の値を返します。
*/
inline uint32_t GetPsnFromSdCardCid(const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    return ((static_cast<uint32_t>(pSdCardCid[5]) << 24) | (static_cast<uint32_t>(pSdCardCid[4]) << 16) | (static_cast<uint32_t>(pSdCardCid[3]) << 8) | pSdCardCid[2]);
}

/**
*   @brief      SD カードの CID から MDT の the year code (0 = 2000) を取得します。
*
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     MDT の the year code の値を返します。
*/
inline uint8_t GetMdtYearCodeFromSdCardCid(const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    return (((pSdCardCid[1] & 0xF) << 4) | (pSdCardCid[0] & 0xF0) >> 4);
}

/**
*   @brief      SD カードの CID から MDT の the month code (1 = January) を取得します。
*
*   @param[in]  pSdCardCid  nn::fs::GetSdCardCid 関数で取得した CID
*
*   @return     MDT の the month code の値を返します。
*/
inline uint8_t GetMdtMonthCodeFromSdCardCid(const uint8_t* pSdCardCid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSdCardCid);
    return (pSdCardCid[0] & 0xF);
}

/**
*   @brief      SD カードの User Area のバイトサイズを取得します。
*
*   @param[out] outValue   取得された User Area のバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result GetSdCardUserAreaSize(int64_t* pOutValue) NN_NOEXCEPT;

/**
*   @brief      SD カードの Protected Area のバイトサイズを取得します。
*
*   @param[out] outValue   取得された Protected Area のバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result GetSdCardProtectedAreaSize(int64_t* pOutValue) NN_NOEXCEPT;

/**
*   @brief      SD カードをフォーマットします。
*               SD カードがマウント状態で呼ばれた場合には、そのマウント状態は無効になります。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*/
Result FormatSdCard() NN_NOEXCEPT;

/**
*   @brief      SD カードがフォーマット可能かテストします。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*/
Result FormatSdCardDryRun() NN_NOEXCEPT;

/**
*   @brief      暗号化のシード値を表す構造体です。
*/
struct EncryptionSeed {
public:
    static const int Size = 16;

public:
    char value[Size];
};
NN_STATIC_ASSERT(std::is_pod<EncryptionSeed>::value);

/**
*   @brief      SD カード上のシステムセーブデータ・コンテンツに適用される暗号化のシード値を設定します。
*               SD カードに対する MountContentStorage(), CreateSystemSaveData(), MountSystemSaveData() を実行する前に設定する必要があります。
*
*   @param[in]  seed   暗号化のシード値
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*/
Result SetSdCardEncryptionSeed(const EncryptionSeed& seed) NN_NOEXCEPT;

/**
*   @brief      ns が SD カードが使用できる状態かどうかをセットします。
*               ns 以外からの呼び出しは禁止です。
*
*   @param[in]  isAccessible   SD カードが使用できる状態かどうかのフラグ
*
*/
void SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT;

/**
*   @brief      ns が SD カードが使用できる状態かどうかを取得します。
*               fs 外からの呼び出しは禁止です。
*
*/
bool IsSdCardAccessible() NN_NOEXCEPT;

namespace detail {

Result SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT;

}

}} // namespace nn::fs
