﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace fs {

/**
* @brief 本体で動的に設定されるセーブデータの識別子です。
*/
typedef uint64_t SaveDataId;

/**
* @brief 静的に設定されるセーブデータの識別子です。
*/
typedef uint64_t StaticSaveDataId;

/**
* @brief システムセーブデータの識別子です。
*/
typedef StaticSaveDataId SystemSaveDataId;

/**
* @brief システム BCAT セーブデータの識別子です。
*/
typedef SystemSaveDataId SystemBcatSaveDataId;

/**
* @brief セーブデータの配置場所の識別子です。
*/
enum class SaveDataSpaceId : uint8_t
{
    System,    //!< 本体内蔵メモリの System 領域
    User,      //!< 本体内蔵メモリの User 領域
    SdSystem,  //!< SD カードの System 領域
    Temporary, //!< 一時ストレージ
    SdUser,    //!< SD カードの User 領域

    ProperSystem = 100,
    SafeMode,
};

/**
* @brief セーブデータの種類です。
*/
enum class SaveDataType : uint8_t
{
    System,    //!< システムセーブデータです。
    Account,   //!< ユーザーアカウントに紐づくアプリケーションセーブデータです。
    Bcat,      //!< アプリケーションの BCAT 配信データの保存に利用するセーブデータです。
    Device,    //!< アプリケーション本体セーブデータです。
    Temporary, //!< 一時ストレージです。
    Cache,     //!< キャッシュストレージです。
    SystemBcat,//!< システムの BCAT 配信データの保存に利用するセーブデータです。
};

/**
    @brief ユーザー識別子
    @details
        ユーザー毎のセーブデータを指定するための識別子です。
        nn::account::Uid からキャストした値を利用してください。
*/
struct UserId
{
    Bit64 _data[2];
};


enum class SaveDataRank : uint8_t
{
    Primary = 0,
    Secondary,
};

enum SaveDataEnumerateOption : uint8_t
{
    SaveDataEnumerateOption_AllRank = 1 << 0, //!< rank != primary のエントリを含めて列挙します。
};

inline bool operator < (const UserId& lhs, const UserId& rhs) NN_NOEXCEPT
{
    if (lhs._data[0] < rhs._data[0])
    {
        return true;
    }
    else if (lhs._data[0] == rhs._data[0] && lhs._data[1] < rhs._data[1])
    {
        return true;
    }
    return false;
}

inline bool operator == (const UserId& lhs, const UserId& rhs) NN_NOEXCEPT
{
    return lhs._data[0] == rhs._data[0] && lhs._data[1] == rhs._data[1];
}

inline bool operator != (const UserId& lhs, const UserId& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

const nn::ncm::ApplicationId InvalidApplicationId = { 0x00ull };
const nn::ncm::ProgramId InvalidProgramId = InvalidApplicationId;
const SystemSaveDataId InvalidSystemSaveDataId = 0x00ull;
const UserId InvalidUserId = {{0x00ull, 0x00ull}};

const int64_t BcatOwnerId = 0x010000000000000CULL;

/**
* @brief セーブデータ情報を表す構造体です。
*/
struct SaveDataInfo
{
    SaveDataId             saveDataId;          //!< セーブデータの識別子です。
    SaveDataSpaceId        saveDataSpaceId;     //!< セーブデータの配置場所の識別子です。
    SaveDataType           saveDataType;        //!< セーブデータの種類です。
    uint8_t                padding1[6];
    UserId                 saveDataUserId;      //!< セーブデータに紐づくユーザーの識別子です。セーブデータがユーザーアカウントに紐づかない場合は nn::fs::InvalidUserId が格納されます。
    SystemSaveDataId       systemSaveDataId;    //!< システムセーブデータの識別子です。システムセーブデータでない場合は nn::fs::InvalidSystemSaveDataId が格納されます。
    nn::ncm::ApplicationId applicationId;       //!< セーブデータに紐づくアプリケーションまたはプログラムの識別子です。
    int64_t                saveDataSize;        //!< セーブデータのファイルシステム上での占有バイトサイズです。
    uint16_t               index;               //!< セーブデータのインデックスです。
    SaveDataRank           rank;                //!< 内部実装用のメンバです。
    uint8_t                reserved[37];
};
static_assert(sizeof(SaveDataInfo) == 96, "sizeof(nn::fs::SaveDataInfo) must be 96.");
NN_STATIC_ASSERT(std::is_pod<SaveDataInfo>::value);

/**
* @brief セーブデータに設定するフラグです。
*/
enum SaveDataFlags : uint32_t
{
    SaveDataFlags_KeepAfterResettingSystemSaveData = 1 << 0,                    //!< 本体初期化で削除されないようにします。
    SaveDataFlags_KeepAfterRefurbishment = 1 << 1,                              //!< リファービッシュで削除されないようにします。
    SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData = 1 << 2, //!< ユーザーセーブデータを残す本体初期化で削除されないようにします。
};

// --------------------------------------------------------------------------------------------
// 以下、内部実装用定義

enum class SaveDataMetaType : uint8_t
{
    None = 0,
    Thumbnail,
    ExtensionContext
};

struct SaveDataMetaInfo
{
    uint32_t size;
    SaveDataMetaType type;
    uint8_t reserved[11];
};
static_assert(sizeof(SaveDataMetaInfo) == 16, "sizeof(nn::fs::SaveDataMetaInfo) must be 16.");
NN_STATIC_ASSERT(std::is_pod<SaveDataMetaInfo>::value);

struct SaveDataCreationInfo
{
    int64_t         size;
    int64_t         journalSize;
    int64_t         blockSize;
    Bit64           ownerId;
    uint32_t        flags;
    SaveDataSpaceId spaceId;
    bool            isPseudoSaveFs; // will be deprecated
    uint8_t         reserved[23];
};
static_assert(sizeof(SaveDataCreationInfo) == 64, "sizeof(nn::fs::SaveDataCreationInfo) must be 64.");
NN_STATIC_ASSERT(std::is_pod<SaveDataCreationInfo>::value);


struct SaveDataAttribute
{
    nn::ncm::ProgramId       programId;
    nn::fs::UserId           userId;
    nn::fs::StaticSaveDataId staticSaveDataId;
    nn::fs::SaveDataType     type;
    nn::fs::SaveDataRank     rank;
    uint16_t                 index;
    uint8_t                  reserved2[28];

    bool operator < (const SaveDataAttribute& rhs) const NN_NOEXCEPT
    {
        if (programId.value < rhs.programId.value)
        {
            return true;
        }
        else if (programId.value == rhs.programId.value &&
                 type < rhs.type)
        {
            return true;
        }
        else if (programId.value == rhs.programId.value &&
                 type == rhs.type &&
                 userId < rhs.userId)
        {
            return true;
        }
        else if (programId.value == rhs.programId.value &&
                 type == rhs.type &&
                 userId == rhs.userId &&
                 staticSaveDataId < rhs.staticSaveDataId)
        {
            return true;
        }
        else if(programId.value == rhs.programId.value &&
                type == rhs.type &&
                userId == rhs.userId &&
                staticSaveDataId == rhs.staticSaveDataId &&
                index < rhs.index)
        {
            return true;
        }
        else if (programId.value == rhs.programId.value &&
                 type == rhs.type &&
                 userId == rhs.userId &&
                 staticSaveDataId == rhs.staticSaveDataId &&
                 index == rhs.index &&
                 rank < rhs.rank)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    bool operator == (const SaveDataAttribute& rhs) const NN_NOEXCEPT
    {
        return
            programId.value == rhs.programId.value &&
            type == rhs.type &&
            userId == rhs.userId &&
            staticSaveDataId == rhs.staticSaveDataId &&
            index == rhs.index &&
            rank == rhs.rank;
    }

    static SaveDataAttribute Make(nn::ncm::ProgramId programId, nn::fs::SaveDataType type, nn::fs::UserId userId, nn::fs::StaticSaveDataId staticSaveDataId, uint16_t index, SaveDataRank rank) NN_NOEXCEPT
    {
        SaveDataAttribute key;
        std::memset(&key, 0x0, sizeof(SaveDataAttribute));
        key.programId.value = programId.value;
        key.type = type;
        key.userId = userId;
        key.staticSaveDataId = staticSaveDataId;
        key.index = index;
        key.rank = rank;
        return key;
    }

    static SaveDataAttribute Make(nn::ncm::ProgramId programId, nn::fs::SaveDataType type, nn::fs::UserId userId, nn::fs::StaticSaveDataId staticSaveDataId, uint16_t index) NN_NOEXCEPT
    {
        return Make(programId, type, userId, staticSaveDataId, index, SaveDataRank::Primary);
    }

    static SaveDataAttribute Make(nn::ncm::ProgramId programId, nn::fs::SaveDataType type, nn::fs::UserId userId, nn::fs::StaticSaveDataId staticSaveDataId) NN_NOEXCEPT
    {
        return Make(programId, type, userId, staticSaveDataId, 0, SaveDataRank::Primary);
    }

};
static_assert(sizeof(SaveDataAttribute) == 64, "sizeof(nn::fs::SaveDataAttribute) must be 64.");

struct SaveDataExtraData
{
    nn::fs::SaveDataAttribute attribute;
    Bit64                     ownerId;
    int64_t                   timeStamp;
    uint32_t                  flags;
    uint8_t                   padding1[4];
    int64_t                   availableSize;
    int64_t                   journalSize;
    int64_t                   commitId;
    uint8_t                   reserved[400];
};
static_assert(sizeof(SaveDataExtraData) == 512, "sizeof(nn::fs::SaveDataExtraData) must be 512.");

struct SaveDataFilter
{
    bool                   filterApplicationId;
    bool                   filterSaveDataType;
    bool                   filterSaveDataUserId;
    bool                   filterStaticSaveDataId;
    bool                   filterIndex;
    uint8_t                option; //!< SaveDataEnumerateOption のビット和
    uint8_t                padding1[2];
    SaveDataAttribute      attribute;

    static SaveDataFilter Make(util::optional<Bit64> applicationId, util::optional<nn::fs::SaveDataType> type, util::optional<nn::fs::UserId> userId, util::optional<nn::fs::StaticSaveDataId> staticSaveDataId, util::optional<uint16_t> index, uint8_t option) NN_NOEXCEPT
    {
        SaveDataFilter filter;
        std::memset(&filter, 0x0, sizeof(SaveDataFilter));

        if (applicationId)
        {
            filter.filterApplicationId = true;
            filter.attribute.programId.value = applicationId.value();
        }
        if (type)
        {
            filter.filterSaveDataType = true;
            filter.attribute.type = type.value();
        }
        if (userId)
        {
            filter.filterSaveDataUserId = true;
            filter.attribute.userId = userId.value();
        }
        if (staticSaveDataId)
        {
            filter.filterStaticSaveDataId = true;
            filter.attribute.staticSaveDataId = staticSaveDataId.value();
        }
        if (index)
        {
            filter.filterIndex = true;
            filter.attribute.index = index.value();
        }
        filter.option = option;
        return filter;
    }

    static SaveDataFilter Make(util::optional<Bit64> applicationId, util::optional<nn::fs::SaveDataType> type, util::optional<nn::fs::UserId> userId, util::optional<nn::fs::StaticSaveDataId> staticSaveDataId, util::optional<uint16_t> index) NN_NOEXCEPT
    {
        return Make(applicationId, type, userId, staticSaveDataId, index, 0);
    }

};
static_assert(sizeof(SaveDataFilter) == 72, "sizeof(nn::fs::SaveDataFilter) must be 72.");
NN_STATIC_ASSERT(std::is_pod<SaveDataFilter>::value);

//! ハッシュのソルト
struct HashSalt
{
    static const auto Size = 32; //!< ソルトのサイズ

    char value[Size]; //!< ソルト
};
NN_STATIC_ASSERT(std::is_pod<HashSalt>::value);

typedef nn::util::optional<HashSalt> SaveDataHashSalt;

/**
* @brief   キャッシュストレージの保存対象メディア
*/
enum CacheStorageTargetMedia : int32_t
{
    CacheStorageTargetMedia_Any,    //!< 使用中のメディア全対象
    CacheStorageTargetMedia_Nand,   //!< 本体メモリのみを対象
    CacheStorageTargetMedia_Sd,     //!< SDカードのみを対象
    CacheStorageTargetMedia_Usb     //!< USBストレージのみを対象
};

}}
