﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  リソースデータ（ROM）関連のプライベート API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_ApplicationId.h>

namespace nn { namespace fs {

//! @name リソースデータ（ROM）関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      指定したアプリケーションのリソースデータ（ROM）に対して nn::fs::MountRom() を実行可能かを取得します。
*
*   @param[in] applicationId アプリケーションの識別子
*
*   @return     処理の結果が返ります。
*   @retval     true                            MountRom() が実行可能です。
*   @retval     false                           MountRom() が実行できません。
*/
bool CanMountRom(nn::ApplicationId applicationId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションを指定した nn::fs::MountRom のファイルシステムキャッシュに必要なサイズを取得します。
*
*   @param[out] pOutValue     ファイルシステムキャッシュに必要なサイズ
*   @param[in]  applicationId アプリケーションの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess        成功しました。
*/
Result QueryMountRomCacheSize(size_t* pOutValue, nn::ApplicationId applicationId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      指定したアプリケーションのリソースデータ（ROM）をマウントします。
*
*   @param[in]  name                      マウント名
*   @param[in]  applicationId             アプリケーションの識別子
*   @param[in]  pFileSystemCacheBuffer    ファイルシステムキャッシュのバッファ
*   @param[in]  fileSystemCacheBufferSize バッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess         成功しました。
*
*   @pre
*               - name が @link mount_name 有効なマウント名 @endlink である。
*               - nn::fs::CanMountRom(applicationId) == true
*               - マウント済みのリソースデータ（ROM）の数が 9 個以下である。
*/
Result MountRom(const char* name, nn::ApplicationId applicationId, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize) NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
