﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_File.h>

namespace nn { namespace fs {

// QueryRange() で取得した情報を格納する構造体
struct QueryRangeInfo
{
    void Clear() NN_NOEXCEPT
    {
        this->aesCtrKeyTypeFlag = 0;
        this->speedEmulationTypeFlag = 0;
        memset(this->_reserved, 0, sizeof(this->_reserved));
    }

    void Merge(QueryRangeInfo info) NN_NOEXCEPT
    {
        this->aesCtrKeyTypeFlag |= info.aesCtrKeyTypeFlag;
        this->speedEmulationTypeFlag |= info.speedEmulationTypeFlag;
    }

    int32_t aesCtrKeyTypeFlag;      // 使用する AES 鍵の種別（AesCtrKeyTypeFlag のビット和）
                                    // ファイルエントリや階層ハッシュ、インダイレクトテーブルの鍵は考慮されません。
                                    // 鍵を使用しないファイル（ホスト上のファイルなど）ではフラグの値は 0 となります。

    int32_t speedEmulationTypeFlag; // 有効になっている速度エミュレーションの種別（SpeedEmulationTypeFlag のビット和）

    char _reserved[56];
};

NN_STATIC_ASSERT(std::is_pod<QueryRangeInfo>::value);
NN_STATIC_ASSERT(sizeof(QueryRangeInfo) == 64); // sf でそのまま引数にできるサイズ

typedef QueryRangeInfo FileQueryRangeInfo;
typedef QueryRangeInfo StorageQueryRangeInfo;

enum class AesCtrKeyTypeFlag : int32_t
{
    InternalKeyForSoftwareAes = 1 << 0,
    InternalKeyForHardwareAes = 1 << 1,
    ExternalKeyForHardwareAes = 1 << 2
};

enum class SpeedEmulationTypeFlag : int32_t
{
    // HardwareAesSpeedEmulationEnabled  = 1 << 0, // HW AES 速度エミュレーションは廃止されたので欠番
    StorageSpeedEmulationEnabled         = 1 << 1,
    //InternalStorageContextEnabled      = 1 << 2, // External 細分化により欠番→ MmcStorageContextEnabled
    //ExternalStorageContextEnabled      = 1 << 3, // External 細分化により欠番→ SdCard/GameCard/UsbStorageContextEnabled
    MmcStorageContextEnabled             = 1 << 4,
    SdCardStorageContextEnabled          = 1 << 5,
    GameCardStorageContextEnabled        = 1 << 6,
    UsbStorageContextEnabled             = 1 << 7,
    //UnspecifiedStorageContextEnabled   = 1 << 8, // StorageFlag_Unspecified 廃止により欠番
};

//--------------------------------------------------------------------------
/**
*   @brief      ファイルの指定範囲にアクセスする際に使用する機能の情報を取得します。
*
*   @param[out] outValue   取得した情報を格納する構造体
*   @param[in]  handle     ファイルのハンドル
*   @param[in]  offset     取得する範囲の開始オフセット
*   @param[in]  size       取得する範囲のサイズ
*
*   @return     処理の結果が返ります。
*/
Result QueryRange(QueryRangeInfo* outValue, FileHandle handle, int64_t offset, int64_t size) NN_NOEXCEPT;



}} // namespace nn::fs
