﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ファイルシステムに関連するファイルシステム プライベート API の宣言
*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_FileSystem.h>

namespace nn { namespace fs {

namespace fsa
{
class IFile;
}

//--------------------------------------------------------------------------
/**
*   @brief      ファイル作成時のオプション指定フラグです。
*
*   @details
*       ファイルを作成する際のオプションを指定します。
*       必要に応じてビットごとの OR で複数指定してください。
*/
    enum CreateFileOptionFlag
    {
        CreateFileOptionFlag_BigFile = 1 << 0,          //!< 4GB 以上になりうるファイルを fat 上に作成する場合に指定するフラグです。
    };

//--------------------------------------------------------------------------
/**
*   @brief      ディレクトリオープン時のモードを指定するフラグです。
*
*   @details
*       ディレクトリを開く際のモードを指定します。
*/
    enum OpenDirectoryModePrivate
    {
        OpenDirectoryModePrivate_NotRequireFileSize = 1 << 31,    //!< ファイルサイズの取得は行わない。
    };


//! @name ファイルシステム関連プライベート API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      ファイルを作成します。
*
*   @param[in]  path       新規に作成するファイルのパス
*   @param[in]  size       新規に作成するファイルのサイズ
*   @param[in]  option     CreateFileOptionFlag, またはそのビットごとの OR
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultPathNotFound              パスに含まれるディレクトリが存在しません。
*   @retval     ResultPathAlreadyExists         対象ファイルはすでに存在しています。
*   @retval     ResultTargetLocked              対象ファイルはすでに存在しています。
*   @retval     ResultUsableSpaceNotEnough      空き容量が不足しています。
*
*   @pre
*       - path が @link entry_path 有効なパス @endlink である。
*       - size >= 0
*/
Result CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      対象エントリの空き容量を取得します。
*
*   @param[out] outValue    空き容量
*   @param[in]  path        空き容量を取得したいパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          対象エントリが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*       - path が @link entry_path 有効なパス @endlink である。
*/
Result GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      対象エントリの全体容量を取得します。
*
*   @param[out] outValue    全体容量
*   @param[in]  path        全体容量を取得したいパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          対象エントリが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*       - path が @link entry_path 有効なパス @endlink である。
*/
Result GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ConcatenationFileSystem の連結ファイル属性を付与します。
*
*   @param[in]  path        属性を付与する対象ディレクトリのパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          対象エントリが存在しません。
*
*   @pre
*       - path が @link entry_path 有効なパス @endlink である。
*       - path がオープン中でない
*       - 対象ファイルシステムが ConcatenationFileSystem である
*/
Result SetConcatenationFileAttribute(const char* path) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      委譲した IFile に紐づくファイルハンドルを発行します。
*
*   @param[out] outValue    ファイルハンドル
*   @param[in]  file        IFile の実装
*   @param[in]  mode        file に対応するアクセスモード（OpenMode 型の要素、またはそのビット和）
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @details
*       委譲した IFile と内部リソースは nn::fs::CloseFile() を呼び出した際に解放されます。
*/
Result OpenFile(FileHandle* outValue, std::unique_ptr<fsa::IFile>&& file, int mode) NN_NOEXCEPT;


//! @}

}} // namespace nn::fs

