﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  デバイスの挿抜・アクセスエラーシミュレーションで使用する API の宣言
*/

#pragma once

namespace nn { namespace fs {

    enum class SimulatingDeviceType : uint32_t
    {
        eMMC,
        SdCard,
        GameCard
    };

    enum class SimulatingDeviceDetectionMode : uint32_t
    {
        NoSimulation,
        DeviceAttached,
        DeviceRemoved
    };

    enum class SimulatingDeviceAccessFailureEventType : uint32_t
    {
        None,
        AccessTimeoutFailure,
        AccessFailure,
        DataCorruption
    };

    enum SimulatingDeviceTargetOperation : uint32_t
    {
        SimulatingDeviceTargetOperation_Read = 1 << 0,
        SimulatingDeviceTargetOperation_Write = 1 << 1
    };

}} // namespace nn::fs


namespace nn { namespace fs {

//! @name 挿抜・アクセスエラーシミュレーション関連 API
//! @{


/**
*   @brief      ゲームカードの挿抜状態およびイベントを疑似的に指定します
*
*   @param[in]  simulatingDetectionMode    シミュレーションのモード
*   @param[in]  isWithEvent                挿抜イベントを発行するかどうか
*
*   @return     処理の結果が返ります。
*/
Result SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode simulatingDetectionMode, bool isWithEvent) NN_NOEXCEPT;

/**
*   @brief      次回のゲームカードの特定オペレーション発行時にエラーシミュレーションを行うよう指定する
*
*   @param[in]  targetOperationFlag       対象となるオペレーションのフラグ（同時指定可）
*   @param[in]  accessFailureEventType    シミュレートするアクセスエラーの種類
*
*   @return     処理の結果が返ります。
*
*/
Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, fs::SimulatingDeviceAccessFailureEventType accessFailureEventType) NN_NOEXCEPT;

/**
*   @brief      次回のゲームカード read アクセス時にエラーシミュレーションを行うよう指定する
*
*   @param[in]  targetOperationFlag       対象となるオペレーションのフラグ（同時指定可）
*   @param[in]  accessFailureEventType    シミュレートするアクセスエラーの種類
*   @param[in]  isPersistent              アクセスエラーのシミュレーションを継続するかどうか（false なら1回だけ発動）
*
*   @return     処理の結果が返ります。
*
*/
Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, fs::SimulatingDeviceAccessFailureEventType accessFailureEventType, bool isPersistent) NN_NOEXCEPT;

/**
*   @brief      次回のゲームカード read アクセス時にエラーシミュレーションを行うよう指定する
*
*   @param[in]  targetOperationFlag       対象となるオペレーションのフラグ（同時指定可）
*   @param[in]  respondingFailureResult   シミュレートするエラー Result の種類
*   @param[in]  isPersistent              アクセスエラーのシミュレーションを継続するかどうか（false なら1回だけ発動）
*
*   @return     処理の結果が返ります。
*
*/
Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, nn::Result respondingFailureResult, bool isPersistent) NN_NOEXCEPT;

/**
*   @brief      アクセスエラーのシミュレーション指示をクリアする
*
*   @return     処理の結果が返ります。
*
*/
Result ClearGameCardSimulationEvent() NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
