﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs/fs_DbmRomTypes.h>

namespace nn { namespace fs {

/*!
    @brief ROM ファイルシステム用パス操作関数群
*/
namespace RomPathTool
{
    // パスの最大長
    static const uint32_t MaxPathLength = 768;

    static const RomPathChar DotChar = '.';

    // エントリ名
    struct RomEntryName
    {
        size_t length;
        const RomPathChar* path;
    };
    NN_STATIC_ASSERT(std::is_pod<RomEntryName>::value);

    /*!
        @brief 指定した文字が区切り文字かどうかを取得します。

        @param[in] pathChar 文字

        @return 区切り文字かどうか
    */
    inline bool IsSeparator(RomPathChar pathChar) NN_NOEXCEPT
    {
        return (pathChar == StringTraitsRom::DirectorySeparator);
    }

    /*!
        @brief カレントディレクトリを示す文字かどうかを取得します。

        @param[in] directoryName ディレクトリ名

        @return カレントディレクトリかどうか
    */
    inline bool IsCurrentDirectory(const RomEntryName& directoryName) NN_NOEXCEPT
    {
        return ((directoryName.length == 1) && (directoryName.path[0] == DotChar));
    }

    /*!
        @brief カレントディレクトリを示す文字かどうかを取得します。

        @param[in] pDirectoryName ディレクトリ名

        @return カレントディレクトリかどうか

        @pre    pDirectoryName != nullptr
    */
    inline bool IsCurrentDirectory(const RomPathChar* pDirectoryName) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryName);
        return ((pDirectoryName[0] == DotChar) && (pDirectoryName[1] == StringTraitsRom::Nul));
    }

    /*!
        @brief カレントディレクトリを示す文字かどうかを取得します。

        @param[in] pDirectoryName   ディレクトリ名
        @param[in] length           ディレクトリ名の文字数

        @return カレントディレクトリかどうか

        @pre    pDirectoryName != nullptr
    */
    inline bool IsCurrentDirectory(const RomPathChar* pDirectoryName, size_t length) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryName);
        return ((length == 1) && (pDirectoryName[0] == DotChar));
    }

    /*!
        @brief 親ディレクトリを示す文字かどうかを取得します。

        @param[in] directoryName ディレクトリ名

        @return カレントディレクトリかどうか
    */
    inline bool IsParentDirectory(const RomEntryName& directoryName) NN_NOEXCEPT
    {
        return ((directoryName.length == 2)
             && (directoryName.path[0] == DotChar)
             && (directoryName.path[1] == DotChar));
    }

    /*!
        @brief 親ディレクトリを示す文字かどうかを取得します。

        @param[in] pDirectoryName ディレクトリ名

        @return カレントディレクトリかどうか

        @pre    pDirectoryName != nullptr
    */
    inline bool IsParentDirectory(const RomPathChar* pDirectoryName) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryName);

        return ((pDirectoryName[0] == DotChar)
             && (pDirectoryName[1] == DotChar)
             && (pDirectoryName[2] == StringTraitsRom::Nul));
    }

    /*!
        @brief 親ディレクトリを示す文字かどうかを取得します。

        @param[in] pDirectoryName   ディレクトリ名
        @param[in] length           ディレクトリ名の文字数

        @return カレントディレクトリかどうか

        @pre    pDirectoryName != nullptr
    */
    inline bool IsParentDirectory(const RomPathChar* pDirectoryName, size_t length) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryName);

        return ((length == 2)
             && (pDirectoryName[0] == DotChar)
             && (pDirectoryName[1] == DotChar));
    }

    /*!
        @brief パスが等しいかどうかを取得します。

        @pre    pExtraKey1 != nullptr
        @pre    pExtraKey2 != nullptr
    */
    inline bool IsEqualPath(
        const RomPathChar* pExtraKey1, const RomPathChar* pExtraKey2, size_t length
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey1);
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey2);

        return (strncmp(pExtraKey1, pExtraKey2, length) == 0);
    }

    /*!
        @brief ROM エントリー名とファイルパスが等しいかどうかを取得します。

        @param[in] name1    ROM エントリー名
        @param[in] pName2   ファイルパス

        @return ROM エントリー名が等しければ true、そうでなければ false

        @pre    pName2 != nullptr
    */
    inline bool IsEqualName(const RomEntryName& name1, const RomPathChar* pName2) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pName2);

        const size_t len = strnlen(pName2, MaxPathLength);
        if( name1.length != len )
        {
            return false;
        }
        return IsEqualPath(name1.path, pName2, len);
    }

    /*!
        @brief ROM エントリー名が等しいかどうかを取得します。

        @param[in] name1 ROM エントリー名
        @param[in] name2 ROM エントリー名

        @return ROM エントリー名が等しければ true、そうでなければ false
    */
    inline bool IsEqualName(const RomEntryName& name1, const RomEntryName& name2) NN_NOEXCEPT
    {
        if( name1.length != name2.length )
        {
            return false;
        }
        return IsEqualPath(name1.path, name2.path, name1.length);
    }

    /*!
        @brief ROM エントリー名を初期化します。

        @param[in] pName ROM エントリー名

        @pre    pName != nullptr
    */
    inline void InitEntryName(RomEntryName* pName) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pName);
        pName->length = 0;
    }

    /*!
        @brief フルパスを解析し、イテレーションするためのクラスです。
    */
    class PathParser
    {
    public:
        /*!
            @brief コンストラクタです。
        */
        PathParser() NN_NOEXCEPT;

        /*!
            @brief パーサを初期化します。

            @param[in] pFullPath フルパス

            @return 関数の処理結果を返します。

            @retval ResultSuccess               成功しました。
            @retval ResultDbmInvalidPathFormat  不正なパスです。

            @pre    pFullPath != nullptr
        */
        Result Initialize(const RomPathChar* pFullPath) NN_NOEXCEPT;

        /*!
            @brief パーサを破棄します。
        */
        void Finalize() NN_NOEXCEPT;

        /*!
            @brief パスのパースが完了したかどうかを取得します。
        */
        bool IsParseFinished() const NN_NOEXCEPT;

        /*!
            @brief パスがディレクトリだったかどうかを取得します。

                   IsParseFinished が true の時のみ使用できます。
        */
        bool IsDirectoryPath() const NN_NOEXCEPT;

        /*!
            @brief 最上位ディレクトリ部分を取得します。

            @param[out] pOutValue ディレクトリ名

            @return 関数の処理結果を返します。

            @retval ResultSuccess                   成功しました。
            @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。

            @pre    初期化済み
            @pre    pOutValue が有効なメモリを指している。
        */
        Result GetNextDirectoryName(RomEntryName* pOutValue) NN_NOEXCEPT;

        /*!
            @brief 文字列の先頭から次の１節をディレクトリ名として取得します。

            @param[out] pOutValue ディレクトリ名

            @return 関数の処理結果を返します。

            @retval ResultSuccess                   成功しました。
            @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。

            @pre    初期化済み
            @pre    pOutValue が有効なメモリを指している。
        */
        Result GetAsDirectoryName(RomEntryName* pOutValue) const NN_NOEXCEPT;

        /*!
            @brief 文字列の先頭から次の１節をファイル名として取得します。

            @param[out] pOutValue ファイル名

            @return 関数の処理結果を返します。

            @retval ResultSuccess                   成功しました。
            @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。

            @pre    初期化済み
            @pre    pOutValue が有効なメモリを指している。
        */
        Result GetAsFileName(RomEntryName* pOutValue) const NN_NOEXCEPT;

    private:
        const RomPathChar* m_pPreviousBeginPath;    // 解析済みのパス始点へのポインタ
        const RomPathChar* m_pPreviousEndPath;      // 解析済みのパス終端へのポインタ
        const RomPathChar* m_pNextPath;             // 次回解析するパスへのポインタ
        bool m_IsParseFinished;                     // NUL 文字に到着したかどうか
    };

    /*!
        @brief 親ディレクトリ名を抽出します。

               フルパス:"/AAA/BBB/CCC" 開始:"AAA" => 親ディレクトリ:"" (ルートディレクトリ)
               フルパス:"/AAA/BBB/CCC" 開始:"BBB" => 親ディレクトリ:"AAA"
               フルパス:"/AAA/BBB/CCC/." 開始:"." => 親ディレクトリ:"BBB"
               フルパス:"/AAA/BBB/CCC/.." 開始:".." => 親ディレクトリ:"AAA"

        @param[out] pOutValue   親ディレクトリ名
        @param[in]  base        開始ディレクトリ名
                                フルパス内の文字列を指している必要があります。
        @param[in]  pHead       フルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDirectoryUnobtainable ルートディレクトリの親を取得しようとしました。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pHead != nullptr
    */
    Result GetParentDirectoryName(
        RomEntryName* pOutValue, const RomEntryName& base, const RomPathChar* pHead
    ) NN_NOEXCEPT;
}

}}
