﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>

#include <nn/fs/fs_DbmRomTypes.h>
#include <nn/fs/fs_SubStorage.h>

namespace nn { namespace fs {

/*!
    @brief 可変長キーバリューストレージテンプレートクラスです。

           2種類のキー(固定サイズのキーと可変サイズのキー)とバリューの組合わせを記憶します。

           キーバリューエレメントの直後に可変サイズのキ－が配置されます。
*/
template <
    typename TKey,
    typename TValue,
    int MaxExtraSize
>
class KeyValueRomStorageTemplate
{
    //! テストのためTestSuiteに対して実装を公開します
    friend class KeyValueRomStorageTemplateTestSuite;

public:
    typedef TKey Key;
    typedef TValue Value;
    typedef uint32_t Position;                  //!< キーバリューストレージのオフセット
    typedef int64_t IndexBucket;                //!< バケットストレージのインデックス

    /*!
        @brief キーと値の列挙に使用する探索位置指示構造体です。
    */
    struct FindIndex
    {
        IndexBucket bucket;
        Position    pos;
    };
    NN_STATIC_ASSERT(std::is_pod<FindIndex>::value);

public:

    /*!
        @brief バケット数からバケットストレージのサイズを求めます。

        @param[in] countBucket バケット数

        @return 必要なストレージのバイトサイズ
    */
    static int64_t QueryBucketStorageSize(int64_t countBucket) NN_NOEXCEPT
    {
        return countBucket * sizeof(Position);
    }

    /*!
        @brief バケットストレージのサイズからからバケット数を求めます。

        @param[in] size バケットストレージのサイズ

        @return バケット数
    */
    static int64_t QueryBucketCount(int64_t size) NN_NOEXCEPT
    {
        return size / sizeof(Position);
    }

    /*!
        @brief 格納可能なキー数からキーバリューストレージのサイズを求めます。

        @param[in] countKeyValue キーバリュー数

        @return 必要なストレージのバイトサイズ
    */
    static int64_t QueryKeyValueStorageSize(uint32_t countKeyValue) NN_NOEXCEPT
    {
        return countKeyValue * sizeof(StorageElement);
    }

    /*!
        @brief バケットを初期化(フォーマット)します。

               バケットストレージを初期化します。使用する前に一度呼び出します。

        @param[in] bucketStorage    バケットストレージ
        @param[in] countBucket      バケット数(要素数)

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を書き込もうとしました。
        @retval それ以外            ストレージの書き込みで失敗しました。

        @return 関数の処理結果を返します。
    */
    static Result Format(
        SubStorage bucketStorage,
        int64_t countBucket
    ) NN_NOEXCEPT
    {
        // 割り当て領域を初期化します。
        const int64_t pos = KvStorageFreeEntry;
        for( int64_t i = 0; i < countBucket; i++ )
        {
            NN_RESULT_DO(
                bucketStorage.Write(i * sizeof(Position), &pos, sizeof(Position))
            );
        }

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief コンストラクタです。
    */
    KeyValueRomStorageTemplate() NN_NOEXCEPT
        : m_CountBucket(0),
          m_BufBucket(),
          m_BufKeyValue(),
          m_TotalEntrySize(0),
          m_EntryCount(0)
    {
    }

    /*!
        @brief ストレージをマウントします。

        @param[in] bucketStorage    バケットストレージ
        @param[in] countBucket      バケット数(要素数)
        @param[in] keyValueStorage  キーバリューストレージ

        @return 関数の処理結果を返します。

        @retval ResultSuccess   成功しました。

        @pre    countBucket > 0
    */
    Result Initialize(
        SubStorage bucketStorage,
        int64_t countBucket,
        SubStorage keyValueStorage
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(countBucket, 0);

        m_BufBucket = bucketStorage;
        m_CountBucket = countBucket;

        m_BufKeyValue = keyValueStorage;

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_BufBucket = SubStorage();
        m_CountBucket = 0;

        m_BufKeyValue = SubStorage();
    }

    /*!
        @brief キーと値を追加します。

               同名のキーが既に存在する場合、ResultDbmAlreadyExists を返します。

        @param[in] key          キー
        @param[in] hashKey      キーのハッシュ値
        @param[in] pExtraKey    可変長キー
        @param[in] extraSize    可変長キーの長さ
        @param[in] value        値

        @return 関数の処理結果を返します。

        @retval ResultSuccess           成功しました。
        @retval ResultDbmAlreadyExists  すでにキーが登録されています。
        @retval ResultDbmKeyFull        割り当て可能領域はありません。
        @retval ResultOutOfRange        ストレージの範囲外をアクセスしようとしました。
        @retval それ以外                ストレージの読み書きで失敗しました。

        @pre    pExtraKey != nullptr
        @pre    extraSize <= MaxExtraSize
    */
    Result Add(
        const Key& key,
        uint32_t hashKey,
        const void* pExtraKey,
        size_t extraSize,
        const Value& value
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey);
        NN_SDK_REQUIRES_LESS_EQUAL(extraSize, MaxExtraSize);
        Position posAdded = 0;
        return AddInternal(&posAdded, key, hashKey, pExtraKey, extraSize, value);
    }

    /*!
        @brief キーに対する値を取得します。

        @param[out] pOutValue   値
        @param[in]  key         キー
        @param[in]  hashKey     キーのハッシュ値
        @param[in]  pExtraKey   可変長キー
        @param[in]  extraSize   可変長キーの長さ

        @return 関数の処理結果を返します。

        @retval ResultSuccess           成功しました。
        @retval ResultDbmKeyNotFound    バケットに何も入っていません。
        @retval ResultDbmKeyNotFound    一致したキーが見つかりません。
        @retval ResultOutOfRange        ストレージの範囲外を読み込もうとしました。
        @retval それ以外                ストレージの読み込みで失敗しました。

        @pre    pExtraKey != nullptr
        @pre    extraSize <= MaxExtraSize
    */
    Result Get(
        Value* pOutValue,
        const Key& key,
        uint32_t hashKey,
        const void* pExtraKey,
        size_t extraSize
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey);
        NN_SDK_REQUIRES_LESS_EQUAL(extraSize, MaxExtraSize);
        Position pos = 0;
        return GetInternal(&pos, pOutValue, key, hashKey, pExtraKey, extraSize);
    }

    /*!
        @brief キーと値の列挙を開始します。

        @param[out] pOutValue イテレータ

        @pre    pOutValue が有効なメモリを指している。
    */
    void FindOpen(FindIndex* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);

        // イテレータを初期化します。
        pOutValue->bucket = static_cast<IndexBucket>(-1);
        pOutValue->pos = KvStorageFreeEntry;
    }

    /*!
        @brief キーと値を列挙します。

        @param[out]     pOutKey     キー
        @param[out]     pOutValue   値
        @param[in,out]  pFindIndex  イテレータ

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFindKeyFinished    列挙が終わりました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @pre    pOutKey が有効なメモリを指している。
        @pre    pOutValue が有効なメモリを指している。
        @pre    pFindIndex != nullptr
    */
    Result FindNext(Key* pOutKey, Value* pOutValue, FindIndex* pFindIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutKey);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pFindIndex);

        IndexBucket indexBucket = pFindIndex->bucket;
        if( (indexBucket >= m_CountBucket) && (indexBucket != static_cast<IndexBucket>(-1)) )
        {
            // キーの探索は終了しています。
            NN_SDK_ASSERT(indexBucket == m_CountBucket);
            return ResultDbmFindKeyFinished();
        }

        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));

        for( ; ; )
        {
            // KeyValueリストを探索します。
            if( pFindIndex->pos != KvStorageFreeEntry )
            {
                StorageElement storeElement;

                // このエントリーを返します。
                NN_RESULT_DO(ReadKeyValue(&storeElement, pFindIndex->pos));

                NN_SDK_ASSERT(
                    (storeElement.next == KvStorageFreeEntry)
                    || (storeElement.next < keyValueSize)
                );

                // 次に進めておきます。
                // リスト終端時は KvStorageFreeEntry が入ります。
                pFindIndex->pos = storeElement.next;

                // キーと値を返します。
                *pOutKey = storeElement.key;
                *pOutValue = storeElement.value;

                NN_RESULT_SUCCESS;
            }

            // バケットリストを探索します。
            for( ; ; )
            {
                // 使用しているバケットリストを読み込みます。
                indexBucket++;
                if( indexBucket == m_CountBucket )
                {
                    // リスト探索終了を示します。
                    pFindIndex->bucket = indexBucket;
                    pFindIndex->pos = KvStorageFreeEntry;

                    return ResultDbmFindKeyFinished();
                }

                Position pos = 0;
                NN_RESULT_DO(ReadBucket(&pos, indexBucket));
                NN_SDK_ASSERT((pos == KvStorageFreeEntry) || (pos < keyValueSize));

                if( pos != KvStorageFreeEntry )
                {
                    // バケットの先頭を捕まえました。
                    pFindIndex->bucket = indexBucket;
                    pFindIndex->pos = pos;
                    break;
                }
            }

        }
    }

    //! 使用エントリー領域のサイズを取得します。
    int64_t GetTotalEntrySize() const NN_NOEXCEPT
    {
        return m_TotalEntrySize;
    }

    //! フリー領域のサイズを取得します。
    Result GetFreeSize(int64_t* pOutValue) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));
        *pOutValue = keyValueSize - m_TotalEntrySize;
        NN_RESULT_SUCCESS;
    }

    //! エントリー数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_EntryCount;
    }

protected:

    /*!
        @brief キーと値を追加します。

               同名のキーが既に存在する場合、値を更新します。

        @param[out] pOutValue   追加されたストレージ位置
        @param[in]  key         キー
        @param[in]  hashKey     キーのハッシュ値
        @param[in]  pExtraKey   可変長キー
        @param[in]  extraSize   可変長キーの長さ
        @param[in]  value       値

        @return 関数の処理結果を返します。

        @retval ResultSuccess           成功しました。
        @retval ResultDbmAlreadyExists  すでにキーが登録されています。
        @retval ResultDbmKeyFull        割り当て可能領域はありません。
        @retval ResultOutOfRange        ストレージの範囲外をアクセスしようとしました。
        @retval それ以外                ストレージの読み書きで失敗しました。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pExtraKey != nullptr
        @pre    マウント済み
    */
    Result AddInternal(
               Position* pOutValue,
               const Key& key,
               uint32_t hashKey,
               const void* pExtraKey,
               size_t extraSize,
               const Value& value
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey);
        NN_SDK_REQUIRES_GREATER(m_CountBucket, 0);

        {
            Position pos = 0;
            Position posPrevious = 0;
            StorageElement storeElement = {};

            // キーが登録済みかどうかチェックします。
            const Result result = FindInternal(
                &pos,
                &posPrevious,
                &storeElement,
                key,
                hashKey,
                pExtraKey,
                extraSize
            );
            if( result.IsSuccess() )
            {
                // すでにキーが登録されています。
                return ResultDbmAlreadyExists();
            }
            else if( !ResultDbmKeyNotFound::Includes(result) )
            {
                // キーが見つからない、以外のコードは
                // ファイルシステムで発生しています。
                return result;
            }
        }

        // キーが見つからない場合、新規にキーを追加します。

        // フリーリストから要素を一つ得てリンクリストに繋ぎます
        Position pos = 0;
        NN_RESULT_DO(AllocateEntry(&pos, extraSize));

        Position posNext = 0;
        NN_RESULT_DO(LinkEntry(&posNext, pos, hashKey));

        const StorageElement storeElement = {
            key,
            value,
            posNext,
            static_cast<uint32_t>(extraSize)
        };

        // エントリーに書き込みます。
        NN_RESULT_DO(
            WriteKeyValue(&storeElement, pos, pExtraKey, extraSize)
        );

        *pOutValue = pos;

        // エントリー数を更新します。
        m_EntryCount++;

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief キーに対する値を取得します。

        @param[out] pOutPosition    格納されているストレージ位置
        @param[out] pOutValue       値
        @param[in]  key             キー
        @param[in]  hashKey         キーのハッシュ値
        @param[in]  pExtraKey       可変長キー
        @param[in]  extraSize       可変長キーの長さ

        @return 関数の処理結果を返します。

        @retval ResultSuccess           成功しました。
        @retval ResultDbmKeyNotFound    バケットに何も入っていません。
        @retval ResultDbmKeyNotFound    一致したキーが見つかりません。
        @retval ResultOutOfRange        ストレージの範囲外を読み込もうとしました。
        @retval それ以外                ストレージの読み込みで失敗しました。

        @pre    pOutPosition が有効なメモリを指している。
        @pre    pOutValue が有効なメモリを指している。
        @pre    pExtraKey != nullptr
    */
    Result GetInternal(
               Position* pOutPosition,
               Value* pOutValue,
               const Key& key,
               uint32_t hashKey,
               const void* pExtraKey,
               size_t extraSize
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutPosition);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey);

        Position pos = 0;
        Position posPrevious = 0;
        StorageElement storeElement = {};

        NN_RESULT_DO(
            FindInternal(
                &pos,
                &posPrevious,
                &storeElement,
                key,
                hashKey,
                pExtraKey,
                extraSize
            )
        );

        *pOutPosition = pos;
        *pOutValue = storeElement.value;

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief ストレージ位置を用いてキーバリューを取得します。

        @param[out] pOutKey         キー
        @param[out] pOutValue       値
        @param[in]  pos             ストレージ位置

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @pre    pOutKey が有効なメモリを指している。
        @pre    pOutValue が有効なメモリを指している。
    */
    Result GetByPosition(
               Key* pOutKey,
               Value* pOutValue,
               Position pos
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutKey);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        StorageElement storeElement = {};
        NN_RESULT_DO(ReadKeyValue(&storeElement, pos));

        *pOutKey = storeElement.key;
        *pOutValue = storeElement.value;

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief ストレージ位置を用いてキーバリューを取得します。

        @param[out] pOutKey         キー
        @param[out] pOutValue       値
        @param[out] pOutExtraKey    可変長キー
        @param[out] pOutExtraSize   可変長キーの長さ
        @param[in]  pos             ストレージ位置

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @pre    pOutKey が有効なメモリを指している。
        @pre    pOutValue が有効なメモリを指している。
        @pre    pOutExtraKey が有効なメモリを指している。
        @pre    pOutExtraSize が有効なメモリを指している。
    */
    Result GetByPosition(
               Key* pOutKey,
               Value* pOutValue,
               void* pOutExtraKey,
               size_t* pOutExtraSize,
               Position pos
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutKey);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pOutExtraKey);
        NN_SDK_REQUIRES_NOT_NULL(pOutExtraSize);

        StorageElement storeElement = {};
        NN_RESULT_DO(ReadKeyValue(&storeElement, pOutExtraKey, pOutExtraSize, pos));

        *pOutKey = storeElement.key;
        *pOutValue = storeElement.value;

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief ストレージ位置を用いてバリューを更新します。

        @param[in] pos      ストレージ位置
        @param[in] value    値

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @return 関数の処理結果を返します。
    */
    Result SetByPosition(
        Position pos, const Value& value
    ) NN_NOEXCEPT
    {
        StorageElement storeElement;
        NN_RESULT_DO(ReadKeyValue(&storeElement, pos));
        storeElement.value = value;
        return WriteKeyValue(&storeElement, pos, nullptr, 0);
    }

private:
    //! アドレスとして無効な値をフリーエントリーとして扱います。
    static const Position KvStorageFreeEntry = 0xFFFFFFFF;

    /*!
        @brief キーバリューストレージのエントリーです。
    */
    struct StorageElement
    {
        Key       key;                     //!< キー
        Value     value;                   //!< 実データへのインデックス
        Position  next;                    //!< 同じハッシュキーを共有する次のデータへのオフセット
        uint32_t  size;                    //!< 拡張キーのサイズ
    };
    NN_STATIC_ASSERT(std::is_pod<StorageElement>::value);

private:

    //! ハッシュ値からバケット位置を取得します。
    IndexBucket HashToBucket(uint32_t hashKey) const NN_NOEXCEPT
    {
        return hashKey % m_CountBucket;
    }

    /*!
        @brief キーを指定し、対応する値セットとストレージ位置を取得します。(内部用)

        @param[out] pOutPosition            得られたストレージ位置
        @param[out] pOutPreviousPosition    得られたストレージ位置を指し示すストレージ位置
        @param[out] pOutStoreElement        得られた値
        @param[in]  key                     キー
        @param[in]  hashKey                 キーのハッシュ値
        @param[in]  pExtraKey               可変長キー
        @param[in]  extraSize               可変長キーの長さ

        @return 関数の処理結果を返します。

        @retval ResultSuccess           成功しました。
        @retval ResultDbmKeyNotFound    バケットに何も入っていません。
        @retval ResultDbmKeyNotFound    一致したキーが見つかりません。
        @retval ResultOutOfRange        ストレージの範囲外を読み込もうとしました。
        @retval それ以外                ストレージの読み込みで失敗しました。

        @pre    pOutPosition が有効なメモリを指している。
        @pre    pOutPreviousPosition が有効なメモリを指している。
        @pre    pOutStoreElement が有効なメモリを指している。
        @pre    pExtraKey != nullptr
        @pre    マウント済み
    */
    Result FindInternal(
               Position* pOutPosition,
               Position* pOutPreviousPosition,
               StorageElement* pOutStoreElement,
               const Key& key,
               uint32_t hashKey,
               const void* pExtraKey,
               size_t extraSize
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutPosition);
        NN_SDK_REQUIRES_NOT_NULL(pOutPreviousPosition);
        NN_SDK_REQUIRES_NOT_NULL(pOutStoreElement);
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey);
        NN_SDK_REQUIRES_GREATER(m_CountBucket, 0);

        *pOutPosition = 0;
        *pOutPreviousPosition = 0;

        // バケット位置を取得します。
        const IndexBucket indexBucket = HashToBucket(hashKey);

        // バケットから先頭位置を取得します。
        Position posTop = 0;
        NN_RESULT_DO(ReadBucket(&posTop, indexBucket));
        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));
        NN_SDK_ASSERT((posTop == KvStorageFreeEntry) || (posTop < keyValueSize));

        // バケットが空だった場合
        if( posTop == KvStorageFreeEntry )
        {
            // バケットには何も入っていません。
            return ResultDbmKeyNotFound();
        }

        const auto bufDeleter = [](uint8_t* ptr) NN_NOEXCEPT
        {
            nn::fs::detail::Deallocate(ptr, MaxExtraSize);
        };
        std::unique_ptr<uint8_t, decltype(bufDeleter)> buf(
            reinterpret_cast<uint8_t*>(nn::fs::detail::Allocate(MaxExtraSize)),
            bufDeleter
        );
        NN_RESULT_THROW_UNLESS(buf != nullptr, nn::fs::ResultAllocationMemoryFailedInDbmKeyValueRomStorageTemplate());

        // アローケーションリンクを辿り、エントリーを探します。
        Position pos = posTop;
        for( ; ; )
        {
            // 一つエントリーを得て比較します。
            size_t currExtraSize = 0;
            NN_RESULT_DO(ReadKeyValue(pOutStoreElement, buf.get(), &currExtraSize, pos));

            // キー同士が一致すれば検索終了です。
            if( key.IsEqual(pOutStoreElement->key, pExtraKey, extraSize, buf.get(), currExtraSize) )
            {
                *pOutPosition = pos;
                NN_RESULT_SUCCESS;
            }

            // 次の位置を探します。
            *pOutPreviousPosition = pos;
            pos = pOutStoreElement->next;
            if( KvStorageFreeEntry == pos )
            {
                // 一致したキーが見つかりません。
                return ResultDbmKeyNotFound();
            }
        }

        // ここには到達しないはず。
    }

    /*!
        @brief フリーエントリーからエントリーを確保します。

        @param[out] pOutValue   得られたエントリー位置
        @param[in]  extraSize   可変長キーのサイズ

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultDbmKeyFull    割り当て可能領域はありません。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result AllocateEntry(Position* pOutValue, size_t extraSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        // フリー領域に十分な領域が存在しているかチェックします。
        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));
        if( m_TotalEntrySize + sizeof(StorageElement) + extraSize > static_cast<size_t>(keyValueSize) )
        {
            // 割り当て可能領域はありません。
            return ResultDbmKeyFull();
        }

         // 得られたエントリーを設定します。
        *pOutValue = static_cast<Position>(m_TotalEntrySize);

        // フリー領域からエントリー + サイズ分確保します。
        // 4 バイトアライメントに切り上げます。
        m_TotalEntrySize
            = nn::util::align_up<int64_t>(
                  static_cast<int64_t>(m_TotalEntrySize + sizeof(StorageElement) + extraSize),
                  4
              );

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief エントリーに繋ぎます。

        @param[out] pOutValue           以前の先頭エントリー
        @param[in]  pos                 先頭にするエントリー
        @param[in]  hashKey             キーのハッシュ値

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外をアクセスしようとしました。
        @retval それ以外            ストレージの読み書きで失敗しました。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result LinkEntry(Position* pOutValue, Position pos, uint32_t hashKey) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        // バケット位置を求めます。
        const IndexBucket indexBucket = HashToBucket(hashKey);

        // バケットの内容を取得します。
        Position posNext = 0;
        NN_RESULT_DO(ReadBucket(&posNext, indexBucket));
        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));
        NN_SDK_ASSERT((posNext == KvStorageFreeEntry) || (posNext < keyValueSize));

        // バケットを更新します。
        NN_RESULT_DO(WriteBucket(pos, indexBucket));
        *pOutValue = posNext;

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief バケットストレージから読み込みます。

        @param[out] pOutValue           バケットに書かれていた値
        @param[in]  index               バケットインデックス

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @pre    pOutValue が有効なメモリを指している。
        @pre    index < m_CountBucket
    */
    Result ReadBucket(Position* pOutValue, IndexBucket index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_LESS(index, m_CountBucket);

        const int64_t offset = index * sizeof(Position);
        return m_BufBucket.Read(offset, pOutValue, sizeof(Position));
    }

    /*!
        @brief バケットストレージに書き込みます。

        @param[in] kvStorePosition  バケットに書き込む値
        @param[in] index            バケットインデックス

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を書き込もうとしました。
        @retval それ以外            ストレージの書き込みで失敗しました。

        @pre    index < m_CountBucket
    */
    Result WriteBucket(
        Position kvStorePosition, IndexBucket index
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(index, m_CountBucket);

        const int64_t offset = index * sizeof(Position);
        return m_BufBucket.Write(offset, &kvStorePosition, sizeof(Position));
    }

    /*!
        @brief キーストレージから読み込みます。

        @param[out] pOutElement     キーバリューストレージに書かれていた値
        @param[in]  pos             ストレージ位置

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @pre    pOutElement が有効なメモリを指している。
    */
    Result ReadKeyValue(
                      StorageElement* pOutElement,
                      Position pos
                  ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutElement);

        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));
        NN_SDK_ASSERT(pos < keyValueSize);

        return m_BufKeyValue.Read(pos, pOutElement, sizeof(StorageElement));
    }

    /*!
        @brief キーストレージから読み込みます。

        @param[out] pOutElement     キーバリューストレージに書かれていた値
        @param[out] pOutExtraKey    可変長キーを読み込むバッファ(MaxExtraSizeバイト)
        @param[out] pOutExtraSize   可変長キーのサイズ
        @param[in]  pos             ストレージ位置

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。

        @pre    pOutElement が有効なメモリを指している。
        @pre    pOutExtraKey が有効なメモリを指している。
        @pre    pOutExtraSize が有効なメモリを指している。
    */
    Result ReadKeyValue(
                      StorageElement* pOutElement,
                      void* pOutExtraKey,
                      size_t* pOutExtraSize,
                      Position pos
                  ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutElement);
        NN_SDK_REQUIRES_NOT_NULL(pOutExtraKey);
        NN_SDK_REQUIRES_NOT_NULL(pOutExtraSize);

        NN_RESULT_DO(ReadKeyValue(pOutElement, pos));

        *pOutExtraSize = pOutElement->size;
        if( pOutElement->size > 0 )
        {
            NN_RESULT_DO(
                m_BufKeyValue.Read(pos + sizeof(StorageElement), pOutExtraKey, pOutElement->size)
            );
        }

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief キーストレージに書き込みます。

        @param[in]  pElement    キーバリューストレージに書き込む値
        @param[in]  pos         ストレージ位置
        @param[in]  pExtraKey   可変長キー
        @param[in]  extraSize   可変長キーのサイズ

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を書き込もうとしました。
        @retval それ以外            ストレージの書き込みで失敗しました。

        @pre    pElement != nullptr
        @pre    pExtraKey != nulltptr
        @pre    pos < キーバリューストレージのサイズ
    */
    Result WriteKeyValue(
                      const StorageElement* pElement,
                      Position pos,
                      const void* pExtraKey,
                      size_t extraSize
                  ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pElement);
        NN_SDK_REQUIRES_NOT_NULL(pExtraKey);

        int64_t keyValueSize = 0;
        NN_RESULT_DO(m_BufKeyValue.GetSize(&keyValueSize));
        NN_SDK_REQUIRES_LESS(pos, keyValueSize);

        NN_RESULT_DO(m_BufKeyValue.Write(pos, pElement, sizeof(StorageElement)));

        if( (pExtraKey != nullptr) && (extraSize > 0) )
        {
            NN_RESULT_DO(
                m_BufKeyValue.Write(pos + sizeof(StorageElement), pExtraKey, extraSize)
            );
        }

        NN_RESULT_SUCCESS;
    }

private:
    int64_t m_CountBucket;                  //!< バケットに割り当てる領域のサイズ(要素数)
    SubStorage m_BufBucket;                 //!< バケットストレージ

    SubStorage m_BufKeyValue;               //!< キーバリューストレージ

    int64_t m_TotalEntrySize;               //!< 使用領域サイズ
    uint32_t m_EntryCount;                  //!< エントリー数

};

}}
