﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ファイルシステムコンテキスト関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {

//! @brief アボート指示子
enum class AbortSpecifier
{
    Default,      //! デフォルトの振る舞いに従う
    Abort,        //! アボートする
    ReturnResult, //! Result を返す
};

//! @brief Result に対するアボート挙動を指示するハンドラ
typedef AbortSpecifier(*ResultHandler)(Result result);

//! @brief ファイルシステムコンテキスト
class FsContext
{
public:
    explicit FsContext(ResultHandler handler) NN_NOEXCEPT
        : m_Handler(handler)
    {
    }

    AbortSpecifier HandleResult(Result result) const NN_NOEXCEPT
    {
        return m_Handler(result);
    }

    void SetHandler(ResultHandler handler) NN_NOEXCEPT
    {
        m_Handler = handler;
    }

private:
    ResultHandler m_Handler;
    // TODO: m_Allocator
};


//! @brief デフォルトコンテキストのリザルトハンドラを設定します。 nullptr を指定するとデフォルトのハンドラが設定されます。
void SetDefaultFsContextResultHandler(const ResultHandler handler) NN_NOEXCEPT;


//! @brief カレントスレッドのコンテキストを設定します。 nullptr を指定するとデフォルトのコンテキストが設定されます。
//         設定されている間、 pContext が指すコンテキストは生存している必要があります。
void SetCurrentThreadFsContext(const FsContext* pContext) NN_NOEXCEPT;


//! @brief カレントスレッドのファイルシステムコンテキストを取得します。
const FsContext* GetCurrentThreadFsContext() NN_NOEXCEPT;


//! @brief 生存する間、カレントスレッドでのファイルシステムコンテキストを設定するユーティリティクラス
class ScopedFsContextSetter
{
public:
    explicit ScopedFsContextSetter(const FsContext& context) NN_NOEXCEPT
        : m_pPreviousContext(GetCurrentThreadFsContext())
    {
        SetCurrentThreadFsContext(&context);
    }

    ~ScopedFsContextSetter() NN_NOEXCEPT
    {
        SetCurrentThreadFsContext(m_pPreviousContext);
    }
private:
    const FsContext* const m_pPreviousContext;
};


//! @brief 生存する間、カレントスレッドでの fs の自動アボートを無効化するユーティリティクラス
class ScopedAutoAbortDisabler
{
public:
    ScopedAutoAbortDisabler() NN_NOEXCEPT;

    ~ScopedAutoAbortDisabler() NN_NOEXCEPT
    {
        SetCurrentThreadFsContext(m_pPreviousContext);
    }
private:
    const FsContext* const m_pPreviousContext;
};


}} // namespace nn::fs
