﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace fs {
    class IEventNotifier;
}}

namespace nn { namespace fssrv { namespace sf {
    class IEventNotifier;
}}}

namespace nn { namespace os {
    class SystemEvent;
}}

namespace nn { namespace fs {

    /**
     *   @brief ファイルシステムアクセスエラー検知イベントを取得します。
     *
     *   @details クリアモードが nn::os::EventClearMode_ManualClear のイベントを取得します。@n
     *            このイベントは、nn::fs::OpenAccessFailureDetectionEventNotifier() の実行でファイルシステムアクセスエラー検知対象になった全プロセスに対して、そのいずれかのプロセスがファイルシステムアクセスが再試行可能なエラーを起こし、その処理が待機状態になったことを示す通知を受け取ることができます。@n
     *            nn::fs::IsAccessFailureDetected() によって、どのプロセスが待機中になっているかを確認することができます。
     *            通知を受け取った後の処理については、nn::fs::OpenAccessFailureDetectionEventNotifier() を参照してください。
     *
     *   @param[out] outValue      取得されたシステムイベント
     */
    Result GetAccessFailureDetectionEvent(os::SystemEvent* outValue) NN_NOEXCEPT;

    /**
     *   @brief 指定したプロセスでファイルシステムアクセスエラーが検知されているかを取得します。
     *
     *   @param[in]  processId     対象プロセスの識別子
     *
     *   @return 検知されていた場合 true が返ります。
     */
    bool IsAccessFailureDetected(nn::Bit64 processId) NN_NOEXCEPT;

    /**
     *   @brief ファイルシステムアクセスエラー通知設定のためのインターフェイスを取得します。
     *
     *   @details 通知対象のプロセスの識別子を引数に入力してください。@n
     *            この API で取得したインターフェイスでは、対象プロセスからのファイルシステムアクセスが再試行可能なエラーを起こし、その処理が待機状態になったことを示す通知を受け取ることができます。@n
     *            その通知を受け取った場合、対象プロセスを停止してください。
     *            その後プロセスを再開する前には、nn::fs::ResolveAccessFailure() の呼び出しが成功することを確認するか、
     *            nn::fs::AbandonAccessFailure() を呼び出してください。@n
     *            nn::fs::AbandonAccessFailure() を呼び出した場合、もしくは対象プロセスを終了した場合には、この関数で取得したインターフェイスとそれに紐づけたイベントを破棄してください。
     *
     *   @param[out] outValue      取得されたインターフェイス
     *   @param[in]  processId     対象プロセスの識別子
     *
     *   @pre
     *       - 対象プロセスに対してこの関数を呼び出して取得したインターフェイスが破棄されている、もしくは存在しない
     *
     *   @return     処理の結果が返ります。
     *   @retval     ResultSuccess               成功しました。
     */
    Result OpenAccessFailureDetectionEventNotifier(std::unique_ptr<IEventNotifier>* outValue, nn::Bit64 processId) NN_NOEXCEPT;

    /**
     *   @brief 待機中のファイルシステムアクセスを解決します。
     *
     *   @details 指定したプロセスからリクエストされて待機中のファイルシステムアクセスを再試行します。@n
     *            再試行に失敗した場合は引き続き待機状態になり、対象プロセスに対して nn::fs::OpenAccessFailureDetectionNotifier() で取得したインターフェイスに再度そのことが通知されます。
     *            この関数は待機中のファイルシステムアクセスがない場合にも呼び出すことができます。
     *
     *   @return     なし
     */
    void ResolveAccessFailure(nn::Bit64 processId) NN_NOEXCEPT;

    /**
     *   @brief 待機中のファイルシステムアクセスの解決を取りやめます。
     *
     *   @details 指定したプロセスからリクエストされて待機中のファイルシステムアクセスを再試行し、成否にかかわらずその結果をアクセス元に返します。@n
     *            この関数を呼び出した場合は対象プロセスを再開し、対象プロセスに対して nn::fs::OpenAccessFailureDetectionNotifier() で取得したインターフェイスとそれに紐づけたイベントを破棄してください。@n
     *            この関数は待機中のファイルシステムアクセスがない場合にも呼び出すことができます。
     *            待機中のファイルシステムアクセスがある場合、対象プロセスを終了した後も、この関数を呼び出すまで待機中の処理に関連するオブジェクトが残留します。
     *            これによるリソースリークを防ぐため、nn::fs::OpenAccessFailureDetectionEventNotifier() の呼び出し前や、プロセス終了時などにこの関数を呼び出しておくことを推奨します。
     *
     *   @return     なし
     */
    void AbandonAccessFailure(nn::Bit64 processId) NN_NOEXCEPT;

    // AM 実装用ユーティリティ
    class AccessFailureResolver : public detail::Newable
    {
        friend Result OpenAccessFailureResolver(std::unique_ptr<AccessFailureResolver>*, nn::Bit64) NN_NOEXCEPT;

    public:
        ~AccessFailureResolver() NN_NOEXCEPT
        {
            AbandonAccessFailure(m_ProcessId);
        }

        void Resolve() NN_NOEXCEPT
        {
            return ResolveAccessFailure(m_ProcessId);
        }

        bool IsAccessFailureDetected() NN_NOEXCEPT
        {
            return fs::IsAccessFailureDetected(m_ProcessId);
        }

    private:
        AccessFailureResolver(nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>&& pHandle, nn::Bit64 processId) NN_NOEXCEPT
            : m_pHandle(pHandle),
              m_ProcessId(processId)
        {
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> m_pHandle;
        nn::Bit64 m_ProcessId;
    };

    Result OpenAccessFailureResolver(std::unique_ptr<AccessFailureResolver>* outValue, nn::Bit64 processId) NN_NOEXCEPT;

}}
