﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace fs { namespace detail {
    void* Allocate(size_t size) NN_NOEXCEPT;

    void Deallocate(void* p, size_t size) NN_NOEXCEPT;

    class Deleter
    {
    public:
        Deleter() NN_NOEXCEPT
            : m_Size(0)
        {
        }
        explicit Deleter(size_t size) NN_NOEXCEPT
            : m_Size(size)
        {
        }
        void operator()(void* p) const NN_NOEXCEPT
        {
            nn::fs::detail::Deallocate(p, m_Size);
        }

    private:
        size_t m_Size;
    };

    //! deleter 付の unique_ptr を返します。
    //! テンプレート引数は POD 型に限ります。
    template<typename T>
    std::unique_ptr<T, Deleter> MakeUnique() NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
        size_t allocationSize = sizeof(T);
        return std::unique_ptr<T, Deleter>(
                   reinterpret_cast<T*>(Allocate(allocationSize)),
                   Deleter(allocationSize)
               );
    }

    //! deleter 付の unique_ptr を返します。
    //! テンプレート引数は POD 型に限ります。
    template<typename Array>
    std::unique_ptr<Array, Deleter> MakeUnique(size_t size) NN_NOEXCEPT
    {
        typedef typename std::remove_extent<Array>::type Type;
        typedef typename std::add_pointer<Type>::type TypePointer;

        NN_STATIC_ASSERT(std::is_pod<Array>::value);
        NN_STATIC_ASSERT(std::is_array<Array>::value);
        size_t allocationSize
            = sizeof(Type) * size;
        return std::unique_ptr<Array, Deleter>(
                   reinterpret_cast<TypePointer>(Allocate(allocationSize)),
                   Deleter(allocationSize)
               );
    }
}}}
