﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_DataId.h>

namespace nn { namespace fs { namespace detail {

//--------------------------------------------------------------------------
/**
*   @brief      MountData のファイルシステムキャッシュに必要なサイズを取得します。
*
*   @param[out] pOutValue                       ファイルシステムキャッシュに必要なサイズ
*   @param[in]  dataId                          対象の Data ID
*   @param[in]  storageId                       検索対象のストレージ ID
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultDataCorrupted             ファイルシステムが壊れています。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*/
Result QueryMountDataCacheSize(
    size_t* pOutValue,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId
) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      Data をマウントします。ファイルシステムキャッシュ、ファイルデータキャッシュ共に利用しません。
*
*   @param[in]  name                            マウント名
*   @param[in]  dataId                          対象の Data ID
*   @param[in]  storageId                       検索対象のストレージ ID
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultMountNameAlreadyExists    name で表されるファイルシステムが既に存在しています。
*   @retval     ResultDataCorrupted             ファイルシステムが壊れています。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*   @retval     ResultTargetNotFound            対象が存在しません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*
*   @details    nn::ncm::StorageId::None を指定した場合はすべてのストレージを対象に検索しマウントします。
*
*/
Result MountData(
    const char* name,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId
) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      Data をマウントします。ファイルシステムキャッシュを利用します。
*
*   @param[in]  name                            マウント名
*   @param[in]  dataId                          対象の Data ID
*   @param[in]  storageId                       検索対象のストレージ ID
*   @param[in]  pFileSystemCacheBuffer          ファイルシステムキャッシュのバッファ
*   @param[in]  fileSystemCacheBufferSize       バッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultMountNameAlreadyExists    name で表されるファイルシステムが既に存在しています。
*   @retval     ResultDataCorrupted             ファイルシステムが壊れています。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*   @retval     ResultTargetNotFound            対象が存在しません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - pFileSystemCacheBuffer != nullptr
*
*   @details    nn::ncm::StorageId::None を指定した場合はすべてのストレージを対象に検索しマウントします。
*
*/
Result MountData(
    const char* name,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId,
    void* pFileSystemCacheBuffer,
    size_t fileSystemCacheBufferSize
) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      Data をマウントします。ファイルシステムキャッシュとファイルデータキャッシュを利用します。
*
*   @param[in]  name                            マウント名
*   @param[in]  dataId                          対象の Data ID
*   @param[in]  storageId                       検索対象のストレージ ID
*   @param[in]  pFileSystemCacheBuffer          ファイルシステムキャッシュのバッファ
*   @param[in]  fileSystemCacheBufferSize       バッファのサイズ
*   @param[in]  makeFileDataCacheAvailable      ファイルデータキャッシュを使用可能にするか否か
*   @param[in]  makePathBasedFileDataCacheAvailable 個別ファイルデータキャッシュを使用可能にするか否か
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultMountNameAlreadyExists    name で表されるファイルシステムが既に存在しています。
*   @retval     ResultDataCorrupted             ファイルシステムが壊れています。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*   @retval     ResultTargetNotFound            対象が存在しません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - pFileSystemCacheBuffer != nullptr
*
*   @details    nn::ncm::StorageId::None を指定した場合はすべてのストレージを対象に検索しマウントします。
*
*/
Result MountData(
    const char* name,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId,
    void* pFileSystemCacheBuffer,
    size_t fileSystemCacheBufferSize,
    bool makeFileDataCacheAvailable,
    bool makePathBasedFileDataCacheAvailable
) NN_NOEXCEPT;

}}}
