﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/time.h>
#include <nn/fs/fsa/fs_IFileSystem.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/nn_Allocator.h>

namespace nn { namespace fat {

    namespace detail {

        class NnResultMaker;

        class FatStorageAdapter
        {
        public:
            FatStorageAdapter() NN_NOEXCEPT : m_pBaseStorage(nullptr), m_LastErrorResult(ResultSuccess()), m_FormatErrorResult(ResultSuccess()) {}

            void Initialize(fs::IStorage* pBaseStorage) NN_NOEXCEPT
            {
                m_pBaseStorage = pBaseStorage;
            }

            void Initialize(std::shared_ptr<fs::IStorage> pBaseStorage) NN_NOEXCEPT
            {
                m_pSharedBaseStorage = std::move(pBaseStorage);
                m_pBaseStorage = m_pSharedBaseStorage.get();
            }

            Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_NOT_NULL(m_pBaseStorage);
                Result result = m_pBaseStorage->Read(offset, buffer, size);
                if (result.IsFailure())
                {
                    m_LastErrorResult = result;
                }
                return result;
            }

            Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_NOT_NULL(m_pBaseStorage);
                Result result = m_pBaseStorage->Write(offset, buffer, size);
                if (result.IsFailure())
                {
                    m_LastErrorResult = result;
                }
                return result;
            }

            Result GetSize(int64_t* outValue) NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_NOT_NULL(m_pBaseStorage);
                Result result = m_pBaseStorage->GetSize(outValue);
                if (result.IsFailure())
                {
                    m_LastErrorResult = result;
                }
                return result;
            }

            Result GetLastErrorResult() NN_NOEXCEPT
            {
                return m_LastErrorResult;
            }

            void SetFormatErrorResult(Result result) NN_NOEXCEPT
            {
                m_FormatErrorResult = result;
            }

            Result GetFormatErrorResult() NN_NOEXCEPT
            {
                return m_FormatErrorResult;
            }

            Result OperateRange(
                void* outBuffer,
                size_t outBufferSize,
                nn::fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize) NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_NOT_NULL(m_pBaseStorage);
                Result result = m_pBaseStorage->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize
                );
                if (result.IsFailure())
                {
                    m_LastErrorResult = result;
                }
                return result;
            }

        private:
            fs::IStorage* m_pBaseStorage;
            std::shared_ptr<fs::IStorage> m_pSharedBaseStorage;
            Result m_LastErrorResult;
            Result m_FormatErrorResult;
        };

        Result FormatDryRun(uint32_t userAreaSectors, uint32_t protectedAreaSectors);
    }

    enum Attribute
    {
        Attribute_Directory = 1 << 0,      //!< ディレクトリ属性
        Attribute_Archive   = 1 << 1,      //!< アーカイブ属性
    };

    struct FatAttribute
    {
        bool isFatSafeEnabled;
        /*
         * trueの場合、FATフォーマットの規格に準拠させます。
         * FAT32の場合は、FS Infoセクタ、及びBackup FS Infoセクタの空きクラスタ情報に無効値(Unknown)を書き込みます。
         * exFATの場合は、Mainブートセクタ、及びBackupブートセクタのPercent In Useフィールドに無効値(Not Available)を書き込みます。
         */
        bool isFatFormatNormalized;
        bool isTimeStampUpdated;
    };

    static const size_t FatFormatWorkBufferSize = 16 * 1024;

    struct FatFormatParam
    {
        bool isSdCard;
        uint32_t protectedAreaSectors; // isSdCard が true 時のみ有効
        Result writeVerifyErrorResult;
        uint8_t* pWorkBuffer;
        size_t workBufferSize;
    };

    static const size_t FatErrorNameMaxLength = 16;

    struct FatError
    {
        int  error;
        int  extraError;
        int  driveId;
        char name[FatErrorNameMaxLength];
        uint8_t reserved[4];
    };
    NN_STATIC_ASSERT(std::is_pod<FatError>::value);
    NN_STATIC_ASSERT(sizeof(FatError) == 32);

    class FatErrorInfoSetter : public nn::fs::detail::Newable
    {
    public:
        FatErrorInfoSetter() NN_NOEXCEPT : m_pFatError(nullptr), m_pMutex(nullptr) {}
        FatErrorInfoSetter(FatError* pFatError, os::Mutex* pMutex, int driveId) NN_NOEXCEPT : m_pFatError(pFatError), m_pMutex(pMutex), m_driveId(driveId) {}
        void SetError(int error, int extraError, const char* name) NN_NOEXCEPT;

    private:
        FatError *m_pFatError;
        os::Mutex* m_pMutex;
        int m_driveId;
    };

    typedef nn::Result(*FatCalendarTimeGetter)(nn::time::CalendarTime*);
    void SetCurrentCalendarTimeGetter(FatCalendarTimeGetter calendarTimeGetter);

    class FatFileSystem : public fs::fsa::IFileSystem, public fs::detail::Newable
    {
    public:
        FatFileSystem() NN_NOEXCEPT;
        virtual ~FatFileSystem() NN_NOEXCEPT;

        /**
            @brief Initialize() に必要なバッファサイズを取得します。
        */
        static size_t GetCacheBufferSize() NN_NOEXCEPT;

        /**
            @brief インスタンスを初期化します。

            @pre
                - 初期化済みでない
                - pBaseStorage は有効な IStorage を指している
                - cacheBuffer は有効なバッファ
                - cacheBufferSize >= GetCacheBufferSize()

            @post
                - 初期化済みである
                - マウント状態でない

            @details cacheBuffer はインスタンスが破棄されるまで参照されます。
        */
        Result Initialize(fs::IStorage* pBaseStorage, void* cacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT;

        /**
            @brief pBaseStorage の所有権を移してインスタンスを初期化します。
        */
        Result Initialize(std::shared_ptr<fs::IStorage> pBaseStorage, void* cacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT;

        /**
            @brief Fatの属性情報を設定してインスタンスを初期化します。
        */
        Result Initialize(fs::IStorage* pBaseStorage, void* cacheBuffer, size_t cacheBufferSize, FatAttribute* pFatAttribute, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT;

        /**
            @brief pBaseStorage の所有権を移してインスタンスを初期化します。
        */
        Result Initialize(std::shared_ptr<fs::IStorage> pBaseStorage, void* cacheBuffer, size_t cacheBufferSize, FatAttribute* pFatAttribute, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT;

        /**
            @brief インスタンスを破棄します。
        */
        void Finalize() NN_NOEXCEPT;

        /**
            @brief Initialize() で与えられた領域を FAT にフォーマットします。

            @pre
                - 初期化済みである
                - 全てのファイル・ディレクトリが開かれていない

            @post
                - Initialize() で与えられた領域 pBaseStorage が FAT フォーマットである
                - マウント状態でない

            @details フォーマットの種別 (FAT12, FAT16, FAT32) は pBaseStorage のサイズから自動的に決定されます。@n
                     すべてのファイル・ディレクトリは削除されます。
        */
        Result Format() NN_NOEXCEPT;

        Result Format(FatFormatParam* pFatFormatParam) NN_NOEXCEPT;

        static bool IsExFatSupported() NN_NOEXCEPT;

        /**
            @brief FAT フォーマットである領域をマウントし、各ファイルシステム API が利用できる状態にします。

            @pre
                - Initialize() で与えられた領域 pBaseStorage が FAT フォーマットである

            @post
                - マウント状態である
        */
        Result Mount() NN_NOEXCEPT;

        /**
            @brief ファイルシステム API です。

            @pre
                - マウント状態である
        */
        virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetEntryType(fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoOpenFile(std::unique_ptr<fs::fsa::IFile>* outValue, const char* path, fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoOpenDirectory(std::unique_ptr<fs::fsa::IDirectory>* outValue, const char* path, fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;

        Result GetAttribute(int* outValue, const char *path) NN_NOEXCEPT;
        Result SetAttribute(const char *path, int attribute) NN_NOEXCEPT;
        Result CreateDirectory(const char *path, int attribute) NN_NOEXCEPT;
        Result GetFileSize(int64_t* outValue, const char *path) NN_NOEXCEPT;

        /**
            @brief MountWithFatSafe(), Format() で利用されるアロケータをセットします。
            @pre
                - pMemoryResource はスレッドセーフなアロケータ
        */
        static void SetAllocatorForFat(nn::MemoryResource* pMemoryResource) NN_NOEXCEPT;

    private:
        os::Mutex m_Mutex;
        class DriveHandle;
        std::unique_ptr<DriveHandle> m_DriveHandle;
        std::unique_ptr<nn::fat::detail::NnResultMaker> m_pNnResultMaker;
        detail::FatStorageAdapter m_FatStorageAdapter;

        Result InitializeImpl(void* cacheBuffer, size_t cacheBufferSize, FatAttribute* pFatAttribute, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT;
        Result ResolvePath(char* outPath, size_t size, const char* path) const NN_NOEXCEPT;
        Result PfGetEntryType(fs::DirectoryEntryType* outValue, const char* pFullPath) NN_NOEXCEPT;
        Result GetEntryTimeStamp(fs::FileTimeStampRaw* outTimeStamp, const char* pFullPath) NN_NOEXCEPT;
        /**
            @brief FAT フォーマットである領域をアンマウントします。
            @pre
                - マウント状態である
        */
        void Unmount() NN_NOEXCEPT;

        /**
            @brief FatSafe 機能を有効として Mount() を実行します。
            @pre
                - SetAllocatorForFatSafe() 実行済み
        */
        Result MountWithFatSafe() NN_NOEXCEPT;

        class FatSafeContext;
        std::unique_ptr<FatSafeContext> m_FatSafeContext;

        class TailBuffer;
        std::unique_ptr<TailBuffer> m_TailBuffer;

        int m_BytesPerCluster;
        bool m_IsMounted;

        FatAttribute m_Attribute;
    };

}}
