﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/eth/sfdl/eth.sfdl.h>
#include <nn/eth/eth_EthTypes.h>
#include <nn/eth/eth_EthClient.h>
#include <nn/eth/eth_Result.public.h>

namespace nn     {
namespace eth    {
namespace client {

/**
 *                    ------------->[Client]<-------------------
 *                    ^                             ^          ^
 *                    |                             |          |
 *      [Adapter Insert/Remove Event]       [Link Up/Down] [Link Up/Down]
 *                    |                             |          |
 *         [class InterfaceGroupHandler]            |          |
 *                                                  |          |
 *                                                  |          |
 *                                    [class InterfaceHandler] |
 *                                                             |
 *                                                 [class InterfaceHandler]
 *
 */

class InterfaceHandler;

class InterfaceGroupHandler
{

private:
    bool                m_Initialized;
    nn::os::SystemEvent m_SystemEvent;
    nn::sf::SharedPointer<nn::eth::sf::IEthInterfaceGroup> m_InterfaceGroupImpl;

public:

    /**
     * @brief Initialize interface group handler, this will set up a system
     *        event which can be used to detect any changes in the interface
     *        group, such as insertion or removal of an adapter.
     *
     * @param[in] eventClearMode
     *            Determines how group event is cleared.
     */
    nn::Result Initialize(nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
     * @brief Finalize interface group handler
     */
    nn::Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Retrieves list of all attached interfaces.
     *
     * @param[out] pInterfaceListOut
     *             List of all attached interfaces
     */
    nn::Result GetInterfaceList(InterfaceList* interfaceListOut) NN_NOEXCEPT;

    /**
     * @brief Helper function to determine removed and added interfaces, based on
     *        previous and current list. Returns added and removed interfaces in
     *        separate structures.
     *
     * @param[in]  pNewList
     *             Current list of interfaces
     * @param[in]  pOldList
     *             Previous list of interfaces
     * @param[out] pAddedList
     *             List of new interfaces
     * @param[out] pRemovedList
     *             List of removed interfaces
     */
    static void ParseInterfaceList(
                    InterfaceList* pNewList,
                    InterfaceList* pOldList,
                    InterfaceList* pAddedList,
                    InterfaceList* pRemovedList) NN_NOEXCEPT;
    /**
     *  @brief Returns number of attached interfaces
     *         (both initialized and non-initialied)
     *
     * @param[out] pInterfaceCount
     *             Number of interfaces
     */
    nn::Result GetInterfaceCount(uint32_t* pInterfaceCount) NN_NOEXCEPT;

    /**
     *  @brief Returns expanded information for interface. Contains
     *         name used for registration with the socket process,
     *         mac address, model/product/unique identifier.
     */
    InterfaceInfo* GetInterfaceInfo() NN_NOEXCEPT;

    /**
     * @brief When event is fired, code of the event is stored and
     *        can be retrieved with this function. Currently
     *        this returna ResultInterfaceGroupChange() or error.
     */
    nn::Result GetResult() NN_NOEXCEPT;

    /**
     * @brief Force the signaling of the system event.
     *        GetResult in this case will return ResutlEventCanceled() code.
     */
    nn::Result Cancel() NN_NOEXCEPT;

    /**
     * @brief When manual clear is used with the system event, this
     *        function can be used to manually reset event.
     */
    void ClearEvent() NN_NOEXCEPT;

    /**
     * @obtain Returns pointer to the system event.
     */
    nn::os::SystemEvent* GetSystemEventPointer() NN_NOEXCEPT;

    InterfaceGroupHandler()  NN_NOEXCEPT;
    ~InterfaceGroupHandler() NN_NOEXCEPT;
};

/**
 *  @brief Encapsulates a single interface, can be used to obtain information
 *         about interface, control interface state - e.g. link, speed, etc,
 *         and obtain events from this interface - e.g. link up/down.
 */
class InterfaceHandler
{

private:
    InterfaceInfo       m_InterfaceInfo;
    bool                m_Initialized;
    nn::os::SystemEvent m_SystemEvent;

    nn::sf::SharedPointer<nn::eth::sf::IEthInterface> m_InterfaceImpl;

public:

    /**
     * @brief Initializes interface handler. Sets up a system event which will
     *        trigger when there is activity on this interface, such as
     *        link loss or autoneg completion.
     *
     * @param[in] pGroupHandler
     *            Pointer to top level handler overseeing all interfaces in this group
     *
     * @param[in] pInfo
     *            Pointer to interface info as provided by the group handler
     *
     * @param[in] eventClearMode
     *            Determines how events are cleared
     */
    nn::Result Initialize(
                    InterfaceGroupHandler* pGroupHandler,
                    InterfaceInfo* pInterfaceInfo,
                    nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
     * @brief Finalize interface handler
     */
    nn::Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Retrieves all supported media types on this interface
     *
     * @param[out] pMediaListOut
     *             Pointer to MediaList structure to store
     *             supported media types are returned
     */
    nn::Result GetMediaList(MediaList* pMediaListOut) NN_NOEXCEPT;

    /**
     * @brief Configures interface
     *
     * @param[in] mediaType
     *            Request media change on this interface,
     *            for example MediaType_AUTO will start autonegotiation.
     */
    nn::Result SetMediaType(MediaType mediaType) NN_NOEXCEPT;

    /**
     * @brief Retrieves current as well as desired state of the interface
     *
     * @param[out] pMediaRequested
     *             Last media type requested in SetMediaType call
     *
     * @param[out] pMediaCurrent
     *             Current state of the media
     *
     * @param[out] pEventCounter
     *             Contains a counter which is incremented at each event on this interface
     */
    nn::Result GetMediaType(
                    MediaType* pMediaRequested,
                    MediaType* pMediaCurrent,
                    uint32_t*  pEventCounter) NN_NOEXCEPT;

    /**
     *  @brief Returns expanded information for this interface -
     *         name, mac address, model/revision/unique identifier.
     */
    InterfaceInfo* GetInterfaceInfo() NN_NOEXCEPT;

    /**
     * @brief Returns name of the interface as registered with socket process
     */
    const char* GetInterfaceName() NN_NOEXCEPT;

    /**
     * @brief When event is triggered, result is stored and can be
     *        obtained using this function. Currently this will
     *        return ResultSuccess(), ResultInterfaceRemoved() or error.
     */
    nn::Result GetResult() NN_NOEXCEPT;

    /**
     * @brief Force event signaling, can be used to unblock waiter.
     *        In this case, GetResult will return ResutlEventCanceled() code.
     */
    nn::Result Cancel() NN_NOEXCEPT;

    /**
     * @brief Clear pending event if manual mode is used
     */
    void ClearEvent() NN_NOEXCEPT;

    /**
     * @obtain Pointer to this event
     */
    nn::os::SystemEvent* GetSystemEventPointer() NN_NOEXCEPT;

    InterfaceHandler()  NN_NOEXCEPT;
    ~InterfaceHandler() NN_NOEXCEPT;
};

}}} // nn::eth
