﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_InterruptEventCommon.h>

namespace nn { namespace dt {

/**
 * @brief       Device Tree の phandle を表す型です。
 */
typedef uint32_t PHandle;

/**
 * @brief       Device Tree のノード名の最大長です。
 * @details     Device Tree の仕様で決められている、ヌル終端を含めたノード名の最大長です。
 */
const size_t MaxNodeNameLength = 32;

/**
 * @brief       Device Tree のプロパティ名の最大長です。
 * @details     Device Tree の仕様で決められている、ヌル終端を含めたプロパティ名の最大長です。
 */
const size_t MaxPropertyNameLength = 32;

/**
 * @brief       Device Tree のノードを表す構造体です。
 * @details     Device Tree のノードを表す構造体です。
 *              dt ライブラリの API で用いるために定義されています。
 *              DDK ユーザーは、この構造体のデフォルトコンストラクタのメンバを使用しないでください。
 */
class Node
{
public:
    /**
     * @brief   デフォルトコンストラクタです。
     */
    Node() NN_NOEXCEPT = default;

    explicit Node(int32_t offset) NN_NOEXCEPT : m_Offset(offset)
    {
    }

    int32_t GetOffset()
    {
        return m_Offset;
    }
private:
    int32_t m_Offset;
};

/**
 * @brief       割り込みが エッジトリガ / レベルトリガ のどちらなのかを示す列挙型です。
 */
enum InterruptTriggerType
{
    InterruptTriggerType_Unspecified,   //!< エッジトリガ / レベルトリガのどちらなのか指定されていません
    InterruptTriggerType_Edge,          //!< エッジトリガです
    InterruptTriggerType_Level,         //!< レベルトリガです
};

/**
 * @brief       レベルトリガの割り込みが active high / active low のどちらなのかを示す列挙型です。
 */
enum InterruptPolarityType
{
    InterruptPolarityType_Unspecified,  //!< active high / active low のどちらなのか指定されていません
    InterruptPolarityType_ActiveHigh,   //!< active high です
    InterruptPolarityType_ActiveLow,    //!< active low です
};

/**
 * @brief       割り込みに関する情報を表す構造体です。
 */
struct InterruptInfo
{
    nn::os::InterruptName name;     //!< 割込みイベントの名前が格納されます。
    InterruptTriggerType trigger;   //!< エッジトリガ / レベルトリガ のどちらなのかの情報が格納されます。
    InterruptPolarityType polarity; //!< active high / active low のどちらなのかの情報が格納されます。
};

}}
