﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree の割り込み関連の機能を提供する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dt/dt_Types.h>

#include <nn/os/os_InterruptEventCommon.h>

namespace nn { namespace dt {

//! @name 割り込み関連の機能を提供する API
//! @{

/**
 * @brief       指定されたノードが扱う割り込みの数を取得します。
 *
 * @param[out]  pOutValue 割り込みの数
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultApiNotSupported}
 * @endretresult
 *
 * @details     pNode に指定したノードが扱う割り込みの数を pOutNode に格納します。
 *              interrupt-parent も参照して処理を行うので、
 *              割り込みコントローラの device tree に不整合があった場合にも
 *              ResultPropertyNotExist などのエラーが返ることに注意してください。
 *              この API は、一部のプラットフォームではサポートされません。
 *              サポートされないプラットフォームで実行した場合、nn::dt::ResultApiNotSupported を返します。
 *
 */
nn::Result GetInterruptCount(int* pOutCount, const Node* pNode) NN_NOEXCEPT;

/**
 * @brief       指定されたノードが扱う割り込みに関する情報を取得します。
 *
 * @param[out]  pOutInfo  割り込みに関する情報
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultApiNotSupported}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     pNode に指定したノードが扱う割り込みに関する情報を取得し、pOutInfo に格納します。
 *              interrupt-parent も参照して処理を行うので、
 *              割り込みコントローラの device tree に不整合があった場合にも
 *              ResultPropertyNotExist などのエラーが返ることに注意してください。
 *              この API は、一部のプラットフォームではサポートされません。
 *              サポートされないプラットフォームで実行した場合、ResultApiNotSupported を返します。
 *              interrupts プロパティに複数の割り込みが定義されていた場合には、ResultSizeIncorrect を返します。
 *              複数の割り込みが定義されている場合は @ref nn::dt::GetInterruptArray() や @ref nn::dt::GetInterruptList() を利用してください。
 *
 */
nn::Result GetInterrupt(nn::dt::InterruptInfo* pOutInfo, const Node* pNode) NN_NOEXCEPT;

/**
 * @brief       指定されたノードが扱う割り込みに関する情報の配列を取得します。
 *
 * @param[out]  pOutArray 割り込みに関する情報の配列
 * @param[in]   count     配列の長さ
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultApiNotSupported}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     pNode に指定したノードが扱う割り込みの一覧を取得し、割り込みに関する情報のリストを pOutList に格納します。
 *              interrupt-parent も参照して処理を行うので、
 *              割り込みコントローラの device tree に不整合があった場合にも
 *              ResultPropertyNotExist などのエラーが返ることに注意してください。
 *              この API は、一部のプラットフォームではサポートされません。
 *              サポートされないプラットフォームで実行した場合、ResultApiNotSupported を返します。
 *              interrupts プロパティ中の割り込みの数が count と一致しなかった場合には、
 *              ResultSizeIncorrect を返します。
 *              count との一致比較を行いたくない場合は、@ref nn::dt::GetInterruptList() を使用してください。
 *
 */
nn::Result GetInterruptArray(nn::dt::InterruptInfo* pOutArray, int count, const Node* pNode) NN_NOEXCEPT;

/**
 * @brief       指定されたノードが扱う割り込みに関する情報の配列を取得します。
 *
 * @param[out]  pOutList  割り込みに関する情報の配列
 * @param[out]  pOutList  pOutList に格納された要素の数
 * @param[in]   maxCount  pOutList に格納できる割り込み番号の最大数
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultApiNotSupported}
 * @endretresult
 *
 * @details     基本的な仕様は @ref nn::dt::GetInterruptArray() と同等ですが、配列長の一致比較を行わない点が異なります。
 *              reg プロパティ中のアドレス・サイズのペアの数が maxCount と一致しなかった場合には、
 *              可能なだけ pOutList にレジスタアドレスをコピーします。
 */
nn::Result GetInterruptList(nn::dt::InterruptInfo* pOutList, int* pOutCount, int maxCount, const Node* pNode) NN_NOEXCEPT;

//! @}

}}
