﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree のデバッグ用 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dt/dt_Types.h>

namespace nn { namespace dt {

//! @name デバッグ用 API
//! @{

/**
 * @brief       ノードのパスを取得します。
 *
 * @param[out]  pOutBuffer  パス文字列を格納するバッファ
 * @param[in]   bufferSize  pOutBuffer のサイズ
 * @param[in]   pNode       ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultBufferTooSmall}
 * @endretresult
 *
 * @details     pNode のパスを pOutBuffer に格納します。pOutBuffer に格納された文字列はヌル終端されます。
 *              バッファサイズが、ヌル終端を含むパス文字列を格納するのに十分な大きさでない場合、ResultBufferTooSmall を返します。
 *              この API はデバッグ用です。製品では使用しないでください。
 *
 */
nn::Result GetNodePath(char* pOutBuffer, size_t bufferSize, const Node* pNode) NN_NOEXCEPT;

/**
 * @brief       ノードの内容をログに出力します。
 *
 * @param[in]   pNode   ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 * @endretresult
 *
 * @details     pNode に含まれるプロパティをログに出力します。
 *              リーフまで再帰的に子ノードを探索して、子ノードのプロパティも出力します。
 *              この API はデバッグ用です。製品では使用しないでください。
 *
 * @internal
 * NN_DETAIL_NO_C_IMPLEMENTATION
 * @endinternal
 *
 */
nn::Result DumpNode(const Node* pNode) NN_NOEXCEPT;

/**
 * @brief       ノードの内容をログに出力します。
 *
 * @param[in]   pNode   ノード
 * @param[in]   depth   子ノード探索の深さ
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 * @endretresult
 *
 * @details     pNode に含まれるプロパティをログに出力します。
 *              再帰的に子ノードを探索して、子ノードのプロパティも出力します。探索の深さは depth で指定します。
 *              depth に 0 を指定すると、pNode のプロパティのみを表示し、子ノードの探索は行いません。
 *              depth に負の値を指定すると、リーフまで探索を行います。
 *              この API はデバッグ用です。製品では使用しないでください。
 *
 */
nn::Result DumpNode(const Node* pNode, int depth) NN_NOEXCEPT;

//! @}

}}
