﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/diag/diag_LogTypes.h>

namespace nn { namespace diag { namespace detail {

template<typename Func>
class FormatStringOutputBufferedDebugPrint
{
public:
    FormatStringOutputBufferedDebugPrint(Func func, char* pBuffer, size_t length, const LogMetaData& logMetaData) NN_NOEXCEPT
        :   m_pBegin(pBuffer),
            // 出力時に '\0' を入れるため pBuffer + length にはしない
            m_pEnd(pBuffer + length - 1),
            m_pCurrent(pBuffer),
            m_Func(func),
            m_LogMetaData(logMetaData),
            m_IsHead(true)
    {
    }

    void Flush(bool isTail) NN_NOEXCEPT
    {
        *m_pCurrent = '\0';
        m_Func(m_LogMetaData, m_pBegin, static_cast<int>(m_pCurrent - m_pBegin), m_IsHead, isTail);
        m_IsHead = false;
        m_pCurrent = const_cast<char*>(m_pBegin);
    }

    static void Output(uintptr_t arg, const char* pCharacters, int count) NN_NOEXCEPT
    {
        reinterpret_cast<FormatStringOutputBufferedDebugPrint*>(arg)->PutCharacters(pCharacters, count);
    }

private:
    void PutCharacters(const char* pCharacters, int count) NN_NOEXCEPT
    {
        while (count > 0)
        {
            if (m_pCurrent == m_pEnd)
            {
                Flush(false);
            }

            const int bufferLeft = static_cast<int>(m_pEnd - m_pCurrent);
            int copyCount = std::min(bufferLeft, count);
            std::memcpy(m_pCurrent, pCharacters, copyCount);

            m_pCurrent += copyCount;
            pCharacters += copyCount;
            count -= copyCount;
        }
    }

    // 出力用バッファの先頭
    char const* m_pBegin;
    // 出力用バッファの末尾
    char const* m_pEnd;
    // 出力用バッファの現在位置
    char* m_pCurrent;
    // 出力用関数のオブジェクト
    Func m_Func;
    // ログのメタデータ
    const LogMetaData& m_LogMetaData;
    // ログの先頭か
    bool m_IsHead;
};

}}}
