﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバサブシステムフレームワークに関する公開ヘッダファイル
 */

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/ddsf/ddsf_CastSafe.h>
#include <nn/ddsf/ddsf_IDevice.h>
#include <nn/ddsf/detail/ddsf_ForEach.h>
#include <nn/ddsf/detail/ddsf_Log.h>

namespace nn { namespace ddsf {

//! @name デバイスドライバサブシステムフレームワーク関連 API
//! @{

/**
* @brief        ドライバサブシステムにおける個別のドライバを表す抽象クラスです。
*
* @detail
*   ドライバサブシステムにおける個別のドライバを表す抽象クラスです。@n
*   このクラスを直接実体化することはできません。@n
*   ドライバサブシステム実装時は、このクラスを public 継承してサブシステム固有のドライバオブジェクトのクラスを実装してください。 @n
*   @n
*   本クラスから派生クラスへの安全なダウンキャストには SafeCastTo<>(), SafeCastToPointer<>() メソッドを使用することを強く推奨します。 @n
*   安全なダウンキャストを使用するには、派生クラスで @ref NN_DDSF_CAST_SAFE_DECL マクロや @ref NN_DDSF_CAST_SAFE_DEFINE マクロを使用した型情報タグの定義が必要です。 @n
*   詳細は @ref ICastSafe のリファレンスを参照してください。
*/
class IDriver : public ICastSafe
{
    NN_DDSF_CAST_SAFE_DECL;

private:
    nn::util::IntrusiveListNode m_ListNode;
    friend class nn::util::IntrusiveListMemberNodeTraits<IDriver, &IDriver::m_ListNode>;

public:
    typedef nn::util::IntrusiveList<
        IDriver, nn::util::IntrusiveListMemberNodeTraits<IDriver, &IDriver::m_ListNode>
    > List;

public:
    //! @name 生成と破棄
    //! @{

    /**
    * @brief    コンストラクタです。
    */
    IDriver() NN_NOEXCEPT
    {
        m_DeviceList.clear();
    }

protected:
    /**
    * @brief    デストラクタです。
    *
    * @detail
    *   デストラクタです。 @n
    *   本クラスは継承を前提としており、直接の実体化や基底クラスに対する delete を禁止するため protected non-virtual としています。
    */
    ~IDriver() NN_NOEXCEPT
    {
        m_DeviceList.clear();
    }

    //! @}

public:

    //! @name ドライバとデバイスの関連づけ
    //! @{

    /**
    * @brief    本オブジェクトを渡されたリストに追加します。
    *
    * @param[in,out]    pList   操作対象のリスト
    *
    * @pre
    *   - pList != nullptr
    */
    void AddTo(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->push_back(*this);
    }

    /**
    * @brief    本オブジェクトを渡されたリストから削除します。
    *
    * @param[in,out]    pList   操作対象のリスト
    *
    * @pre
    *   - pList != nullptr
    */
    void RemoveFrom(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->erase(pList->iterator_to(*this));
    }

    /**
    * @brief    本オブジェクトがリストに追加された状態かどうかを返します。
    *
    * @return   リストに追加された状態かどうか
    */
    bool IsLinkedToList() const NN_NOEXCEPT
    {
        return m_ListNode.IsLinked();
    }

    /**
    * @brief    ドライバにひとつでも登録されたデバイスがあるかどうかを返します。
    *
    * @return   登録されたデバイスがあるかどうか
    */
    bool HasAnyDevice() const NN_NOEXCEPT
    {
        return !m_DeviceList.empty();
    }

    /**
    * @brief    ドライバにデバイスを登録します。
    *
    * @param[in,out]    pDevice   デバイス
    *
    * @pre
    *   - pDevice != nullptr
    *
    * @post
    *   - HasAnyDevice() == true
    */
    void RegisterDevice(IDevice* pDevice) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
        std::lock_guard<decltype(m_DeviceListLock)> lock(m_DeviceListLock);
        pDevice->AttachDriver(this);
        m_DeviceList.push_back(*pDevice);
    }

    /**
    * @brief    ドライバからデバイスの登録を解除します。
    *
    * @param[in,out]    pDevice   デバイス
    *
    * @pre
    *   - pDevice != nullptr
    *
    * @post
    *   - HasAnyDevice() == true
    */
    void UnregisterDevice(IDevice* pDevice) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
        std::lock_guard<decltype(m_DeviceListLock)> lock(m_DeviceListLock);
        m_DeviceList.erase(m_DeviceList.iterator_to(*pDevice));
        pDevice->DetachDriver();
    }

    //! @}

    //! @name デバイスの操作
    //! @{

    // mutex でリストアクセスを守るため、リストを直接返す API は用意せず、代わりに ForEach 機能を提供する

    /**
    * @brief ドライバに対して登録されている全てのデバイスに対して指定したメソッドを呼び出します。（非 const 、リザルトハンドリングあり）
    *
    * @tparam   FuncT   IDevice* を引数に取り nn::Result を返す関数として振る舞える型
    *
    * @param[in] func               呼び出すメソッド
    * @param[in] returnOnFailure     @a func が失敗したときの挙動オプション
    *   returnOnFailure | 動作
    *   -------------- | ------
    *   true           | func のひとつが nn::ResultSuccess 以外を返した時点で以降のメソッド呼び出しを中止する
    *   false          | func のひとつが nn::ResultSuccess 以外を返しても以降のメソッド呼び出しを続ける
    *
    * @return
    *   すべての @a func の呼び出しが nn::ResultSuccess を返したとき、 nn::ResultSuccess を返します。 @n
    *   いずれかの @a func の呼び出しが nn::ResultSuccess 以外のリザルトを返したとき、そのリザルトを返します。 @n
    *   @a returnOnFailure が false で、複数の func 呼び出しが失敗した場合は、最初の失敗のリザルトを返します。
    *
    * @detail
    *   ドライバに対して登録されている全てのデバイスに対して指定したメソッドを呼び出します。 @n
    *   @n
    *   本関数の実行は、内部でデバイスリストへの要素の追加・削除操作と排他制御されます。
    */
    template<typename FuncT>
    nn::Result ForEachDevice(FuncT func, bool returnOnFailure) NN_NOEXCEPT
    {
        return detail::ForEach(&m_DeviceList, func, &m_DeviceListLock, returnOnFailure);
    }

    /**
    * @brief ドライバに対して登録されている全てのデバイスに対して指定したメソッドを呼び出します。（const 、リザルトハンドリングあり）
    *
    * @tparam   FuncT   const IDevice* を引数に取り nn::Result を返す関数として振る舞える型
    *
    * @param[in] func               呼び出すメソッド
    * @param[in] returnOnFailure     @a func が失敗したときの挙動オプション
    *
    * @detail
    *   呼び出すメソッドの引数が const 参照になる以外は非 const 版と同じです。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    template<typename FuncT>
    nn::Result ForEachDevice(FuncT func, bool returnOnFailure) const NN_NOEXCEPT
    {
        return detail::ForEach(&m_DeviceList, func, &m_DeviceListLock, returnOnFailure);
    }

    /**
    * @brief ドライバに対して登録されている全てのデバイスに対して指定したメソッドを呼び出します。（非 const 、リザルトハンドリングなし）
    *
    * @tparam   FuncT   IDevice* を引数に取り bool 評価可能な値を返す関数として振る舞える型
    *
    * @param[in] func   呼び出すメソッド
    *
    * @return   func が true を返した回数
    *
    * @detail
    *   ドライバに対して登録されている全てのデバイスに対して指定したメソッドを呼び出します。 @n
    *   ForEachDevice<FuncT>(FuncT, bool) と異なり、こちらではリザルトのハンドリングがありません。 @n
    *   呼び出された @a func が true を返す限り、続く要素へのメソッド呼び出しを継続します。 @n
    *   @a func が false を返した時点で以降のメソッド呼び出しを中止します。 @n
    *   @n
    *   本関数の実行は、内部でデバイスリストへの要素の追加・削除操作と排他制御されます。
    */
    template<typename FuncT>
    int ForEachDevice(FuncT func) NN_NOEXCEPT
    {
        return detail::ForEach(&m_DeviceList, func, &m_DeviceListLock);
    }

    /**
    * @brief ドライバに対して登録されている全てのデバイスに対して指定したメソッドを呼び出します。（const 、リザルトハンドリングなし）
    *
    * @tparam   FuncT   const IDevice* を引数に取り bool 評価可能な値を返す関数として振る舞える型
    *
    * @param[in] func   呼び出すメソッド
    *
    * @return   func が true を返した回数
    *
    * @detail
    *   呼び出すメソッドの引数が const 参照になる以外は非 const 版と同じです。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    template<typename FuncT>
    int ForEachDevice(FuncT func) const NN_NOEXCEPT
    {
        return detail::ForEach(&m_DeviceList, func, &m_DeviceListLock);
    }

    //! @}

private:
    IDevice::List m_DeviceList;
    mutable nn::os::SdkMutex m_DeviceListLock;
};

//! @}

}} // namespace nn::ddsf
