﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバ向けキャッシュ操作に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/dd/dd_Types.h>

namespace nn { namespace dd {

//! @name キャッシュ操作関連 API
//! @{

/**
 * @brief   データキャッシュを破棄します。
 *
 * @param[in] addr 開始アドレス
 * @param[in] size サイズ
 *
 * @pre
 *  - 指定されたメモリ領域がキャッシュ対象である
 *  - 指定されたメモリ領域が書き込み可能である
 *
 * @details
 *  指定されたメモリ領域のデータキャッシュを破棄します。
 */
void InvalidateDataCache( void* addr, size_t size ) NN_NOEXCEPT;

/**
 * @brief   データキャッシュをメモリに書き出します。
 *
 * @param[in] addr 開始アドレス
 * @param[in] size サイズ
 *
 * @pre
 *  - 指定されたメモリ領域がキャッシュ対象である
 *  - 指定されたメモリ領域が読み込みまたは書き込み可能である
 *
 * @details
 *  指定されたメモリ領域のデータキャッシュをメモリに書き出します。
 */
void StoreDataCache( const void* addr, size_t size ) NN_NOEXCEPT;

/**
 * @brief   データキャッシュをメモリに書き出したのち破棄します。
 *
 * @param[in] addr 開始アドレス
 * @param[in] size サイズ
 *
 * @pre
 *  - 指定されたメモリ領域がキャッシュ対象である
 *  - 指定されたメモリ領域が読み込みまたは書き込み可能である
 *
 * @details
 *  指定されたメモリ領域のデータキャッシュをメモリに書き出したのち破棄します。
 */
void FlushDataCache( const void* addr, size_t size ) NN_NOEXCEPT;


/**
 * @brief   指定されたプロセスのデータキャッシュを破棄します。
 *
 * @param[in] handle 対象プロセスを指すハンドル
 * @param[in] addr   開始アドレス
 * @param[in] size   サイズ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultInvalidHandle}
 *   @handleresult{nn::dd::ResultInvalidMemoryState}
 *   @handleresult{nn::dd::ResultNotSupported}
 * @endretresult
 *
 * @pre
 *  - 指定されたプロセスの対象メモリ領域がキャッシュ対象である
 *  - 指定されたプロセスの対象メモリ領域が書き込み可能である
 *
 * @details
 *  指定されたプロセスの対象メモリ領域のデータキャッシュを破棄します。
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、常に nn::dd::ResultNotSupported が返ります。
 *
 * @platformend
 *
 */
Result InvalidateProcessDataCache( ProcessHandle handle, uint64_t addr, uint64_t size ) NN_NOEXCEPT;

/**
 * @brief   指定されたプロセスのデータキャッシュをメモリに書き出します。
 *
 * @param[in] handle 対象プロセスを指すハンドル
 * @param[in] addr   開始アドレス
 * @param[in] size   サイズ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultInvalidHandle}
 *   @handleresult{nn::dd::ResultInvalidMemoryState}
 *   @handleresult{nn::dd::ResultNotSupported}
 * @endretresult
 *
 * @pre
 *  - 指定されたプロセスの対象メモリ領域がキャッシュ対象である
 *  - 指定されたプロセスの対象メモリ領域が読み込みまたは書き込み可能である
 *
 * @details
 *  指定されたプロセスの対象メモリ領域のデータキャッシュをメモリに書き出します。
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、常に nn::dd::ResultNotSupported が返ります。
 *
 * @platformend
 *
 */
Result StoreProcessDataCache( ProcessHandle handle, uint64_t addr, uint64_t size ) NN_NOEXCEPT;

/**
 * @brief   指定されたプロセスのデータキャッシュをメモリに書き出したのち破棄します。
 *
 * @param[in] handle 対象プロセスを指すハンドル
 * @param[in] addr   開始アドレス
 * @param[in] size   サイズ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultInvalidHandle}
 *   @handleresult{nn::dd::ResultInvalidMemoryState}
 *   @handleresult{nn::dd::ResultNotSupported}
 * @endretresult
 *
 * @pre
 *  - 指定されたプロセスの対象メモリ領域がキャッシュ対象である
 *  - 指定されたプロセスの対象メモリ領域が読み込みまたは書き込み可能である
 *
 * @details
 *  指定されたプロセスの対象メモリ領域のデータキャッシュを
 *  メモリに書き出したのち破棄します。
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、常に nn::dd::ResultNotSupported が返ります。
 *
 * @platformend
 *
 */
Result FlushProcessDataCache( ProcessHandle handle, uint64_t addr, uint64_t size ) NN_NOEXCEPT;

//! @}

}}  // namespace nn::dd
