﻿using Nintendo.Nact;
using Nintendo.Nact.BackEnd;
using Nintendo.Nact.BuildEngine;
using Nintendo.Nact.ClassSystem;
using Nintendo.Nact.Evaluation;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Plugin;
using Nintendo.Nact.Utilities;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using static Nintendo.Nact.Extensions.NactValueExtensions;

namespace SigloNact
{
    [NactBackEndPlugin]
    public class SigloNactBackEndPlugin : INactBackEndPlugin
    {
        private static readonly IEnumerable<System.Reflection.Assembly> BuiltInContainerAssemblies = new[]
        {
            typeof(SigloNactBackEndPlugin).Assembly,
        };

        private bool m_DisposedValue = false;

        public SigloNactBackEndPlugin()
        {
            InitializationUtil.InitializeResourceManagers();
        }

        #region IDisposable Support
        protected virtual void Dispose(bool disposing)
        {
            if (!m_DisposedValue)
            {
                if (disposing)
                {
                }

                m_DisposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion

        public IEnumerable<System.Reflection.Assembly> GetBuiltInContainerAssemblies() => BuiltInContainerAssemblies;

        public IEnumerable<EnvironmentTestResultItem> TestEnvironment()
        {
            return new SigloEnvironmentChecker(SigloNactAppSettings.BootedFromSigloRepository).Check();
        }

        public void ApplyOverrideUICulture(CultureInfo uiCulture)
        {
            InitializationUtil.ApplyOverrideUICulture(uiCulture);
        }

        public IEnumerable<DeleteObsoleteFilesTarget> GetDeleteObsoleteFilesTargets(
            EvaluationState evaluationState,
            BuildTaskTree buildTaskTree)
        {
            var libraryExtensions = new string[]
            {
                ".lib", ".a", ".o",

                // msvc
                // ".pdb",  // pdb ファイルは現状 Results に指定されていない (複数のルールが同一の pdb ファイルを読み書きする問題があるため) ので、削除対象にしない
                // Cafe
                ".dba",
                // horizon
                ".nso", ".nss", ".nro", ".nrs"
            };
            var headerExtensions = new string[] { ".h", ".hpp" };

            var subRootDefinitions = SubRootDefinition.GetSubRootDefinitions(evaluationState, buildTaskTree);
            var outputIncludeDirectories = Utilities.GetDeleteObsoleteFilesTargetsUtil.GetDeleteTargetDirectories(evaluationState, buildTaskTree, "SigloNactOutputIncludeDirectories");
            var toolDirectories = Utilities.GetDeleteObsoleteFilesTargetsUtil.GetDeleteTargetDirectories(evaluationState, buildTaskTree, "SigloNactToolDirectories");
            var resourceDirectories = Utilities.GetDeleteObsoleteFilesTargetsUtil.GetDeleteTargetDirectories(evaluationState, buildTaskTree, "SigloNactResourceDirectories");

            return subRootDefinitions.Select(x => DeleteObsoleteFilesTarget.CreateWithFilter(x.LibrariesDirectory, libraryExtensions, true))
            .Concat(outputIncludeDirectories.Select(x => DeleteObsoleteFilesTarget.CreateWithFilter(x, headerExtensions, false)))
            .Concat(toolDirectories.Select(x => DeleteObsoleteFilesTarget.CreateWithOutFilter(x, false)))
            .Concat(resourceDirectories.Select(x => DeleteObsoleteFilesTarget.CreateWithOutFilter(x, false)));
        }

        public void ExecuteSubcommandHelp(EvaluationState evaluationState, BuildTaskTree buildTaskTree)
        {
            // TORIAEZU: ルールから help 文字列を得る
            // TODO: help サブコマンドを実行するのにルールを評価しなくてよいようにする（ルールメタ情報でユーザオプション等を定義する）
            BuildNode rootNode;
            if (!buildTaskTree.BuildNodeCollection.TryGetNodeByPath(buildTaskTree.RootRulePath, out rootNode))
            {
                throw new InternalLogicErrorException("Root build node cannot be found");
            }
            try
            {
                var ev = new Evaluator(evaluationState, rootNode, rootNode);
                var text = ev.EvaluateVariableReferenceSourceCode("SigloNactHelpText").CheckedCast<string>();
                Nintendo.Nact.Utilities.Log.Message(text);
            }
            catch (ErrorException)
            {
                // TORIAEZU: TestRunner が生成した nact ルールを与えてもエラーにならないように回避
                Nintendo.Nact.Utilities.Log.Message("No help text defined.");
            }
        }

        public void CheckCommandLines(
            EvaluationState evaluationState,
            BuildTaskTree buildTaskTree,
            BuildResult buildResult)
        {
            new SigloWildcardCopyChecker(evaluationState, buildTaskTree, buildResult).Check();
        }

        public IEnumerable<IVersionControlInfo> GetVersionControlInfos(EvaluationState evaluationState, BuildTaskTree buildTaskTree)
        {
            // ToDo: git exrepo の情報も取得する
            FilePath sigloRepositoryRootPath;
            {
                BuildNode rootNode;
                if (!buildTaskTree.BuildNodeCollection.TryGetNodeByPath(buildTaskTree.RootRulePath, out rootNode))
                {
                    throw new InternalLogicErrorException("Root build node cannot be found");
                }
                try
                {
                    var ev = new Evaluator(evaluationState, rootNode, rootNode);
                    sigloRepositoryRootPath = ev.EvaluateVariableReferenceSourceCode("SigloNactRepositoryRootPath").CheckedCast<FilePath>();
                }
                catch (ErrorException)
                {
                    // リポジトリのルートが定義されていなければ空配列を返す
                    return Array.Empty<IVersionControlInfo>();
                }
            }
            return new[] { new GitInfo(sigloRepositoryRootPath) };
        }
    }
}
