﻿using Nintendo.Nact;
using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using Nintendo.Nact.Utilities.ProgramExecution;
using SigloNact.Build;
using SigloNact.Utilities;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using static System.FormattableString;

namespace SigloNact.BuiltIns.ToolChain.Msvc
{
    [NactActionFunctionContainer]
    public static class MsvcActions
    {
        // コンパイル結果が変わるような変更（CommonOptions の変更を含む）を行った場合、バージョンを上げること
        private const int MsvcActionVersion = 1;

        private static IReadOnlyCollection<string> WarningLevelOptions = Enumerable.Range(0, 5).Select(x => Invariant($"/W{x}")).ToArray();

        [NactActionFunction(Version = MsvcActionVersion)]
        public static NactActionResult MsvcCompileAction(
            INactActionContext context,
            MsvcToolChainParameters toolChainParameters,
            MsvcToolPathProvider toolPathProvider,
            FilePath outputObjectFile,
            FilePath outputPdbFile,
            FilePath outputAsmFile,
            FilePath sourceFile,
            string precompiledHeaderMode,
            IReadOnlyCollection<FilePath> includeDirectories,
            IReadOnlyCollection<string> preprocessorMacros,
            IReadOnlyCollection<string> compileOptions,
            FilePath precompiledHeaderSourceFile,
            FilePath precompiledHeaderObjectFile)
        {
            var precompiledHeaderModeParsed = Util.ParseEnum<PrecompiledHeaderMode>(precompiledHeaderMode);

            var helper = context.Helper;

            // TODO: EnsureDirectory するユーティリティ
            // pdb は Results に指定できていないので、アクション側でディレクトリを用意してあげる
            Util.EnsureDirectory(outputPdbFile.Parent.PathString);

            var flags = new List<string>();
            MsvcCommonOptions.GetClOptions(flags, toolChainParameters);
            flags.AddRange(preprocessorMacros.Select(x => MsvcHelper.MakeProprocessorMacroOption(x)));

            switch (precompiledHeaderModeParsed)
            {
                case PrecompiledHeaderMode.None:
                    break;
                case PrecompiledHeaderMode.Use:
                case PrecompiledHeaderMode.Create:
                    flags.Add(Invariant($"/Fp\"{precompiledHeaderObjectFile}\""));
                    flags.Add(Invariant($"/FI\"{precompiledHeaderSourceFile.FileName}\"")); // FIXME: FileName なのか？
                    if (precompiledHeaderModeParsed == PrecompiledHeaderMode.Use)
                    {
                        flags.Add(Invariant($"/Yu\"{precompiledHeaderSourceFile.FileName}\""));
                    }
                    else
                    {
                        flags.Add(Invariant($"/Yc\"{precompiledHeaderSourceFile.FileName}\""));
                    }
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // 入出力ファイル
            flags.Add("/FA");
            flags.Add(Invariant($"/Fa\"{outputAsmFile}\""));
            flags.Add(Invariant($"/Fd\"{outputPdbFile}\""));
            flags.Add(Invariant($"/Fo\"{outputObjectFile}\""));
            flags.Add(Invariant($"\"{sourceFile}\""));
            flags.Add("/c");

            // TORIAEZU: 成果物独自に警告レベルを下げている場合は、デフォルトの警告レベルオプションを抜く
            // TODO: additionalCompileOptions をパースして設定値を得て、デフォルトの設定値と合成する
            // （additionalCompileOptions 中の許可されていないオプションを拒否することにもなる）
            if (compileOptions.Any(x => WarningLevelOptions.Contains(x)))
            {
                flags.RemoveAll(x => WarningLevelOptions.Contains(x));
            }

            flags.AddRange(compileOptions);

            var result = helper.ExecuteProgram(
                toolPathProvider.ClExePath,
                string.Join(" ", flags),
                MakeEnvironmentVariablesForCompile(toolPathProvider, includeDirectories),
                outputModifier: text => ClExeOutputModifier(sourceFile.FileName, text),
                warningDetector: MsvcHelper.WarningDetector,
                errorDetector: MsvcHelper.ErrorDetector);

            if (result.ExitCode != 0)
            {
                // TODO: OutputModifier に渡すときに文字列化してあるのだから、INactActionHelper.ExecuteProgram が文字列 (lines) を返して欲しい
                if (IsTransientError(result, IsClExeTransientError))
                {
                    // TODO: 一過性のエラーを返すよいインターフェース
                    return helper.FinishAsFailure().AsTransientFailure();
                }
                else
                {
                    return helper.FinishAsFailure();
                }
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction(Version = MsvcActionVersion)]
        public static NactActionResult MsvcMakeDependenciesAction(
            INactActionContext context,
            MsvcToolChainParameters toolChainParameters,
            MsvcToolPathProvider toolPathProvider,
            FilePath outputDependenciesFile,
            FilePath sourceFile,
            IReadOnlyCollection<FilePath> includeDirectories,
            IReadOnlyCollection<string> preprocessorMacros)
        {
            var helper = context.Helper;

            var flags = new List<string>();
            flags.AddRange(preprocessorMacros.Select(x => MsvcHelper.MakeProprocessorMacroOption(x)));
            flags.Add("/c");
            flags.Add("/nologo");
            flags.Add("/Fonul");
            flags.Add("/TP");
            flags.Add("/EHsc");
            flags.Add("/showIncludes");
            flags.Add(Invariant($"\"{sourceFile}\""));

            var result = helper.ExecuteProgram(
                toolPathProvider.ClExePath,
                string.Join(" ", flags),
                MakeEnvironmentVariablesForCompile(toolPathProvider, includeDirectories),
                outputModifier: filterOutIncludeLines,
                warningDetector: MsvcHelper.WarningDetector,
                errorDetector: MsvcHelper.ErrorDetector);

            if (result.ExitCode != 0)
            {
                // TODO: OutputModifier に渡すときに文字列化してあるのだから、INactActionHelper.ExecuteProgram が文字列 (lines) を返して欲しい
                if (IsTransientError(result, IsClExeTransientError))
                {
                    // TODO: 一過性のエラーを返すよいインターフェース
                    return helper.FinishAsFailure().AsTransientFailure();
                }
                else
                {
                    return helper.FinishAsFailure();
                }
            }

            helper.WriteAllBytes(outputDependenciesFile, result.StandardOutput);

            return helper.FinishAsSuccess();

            void filterOutIncludeLines(List<string> lines)
            {
                var (_, notIncludeLines) = ConvertShowIncludesUtil.PartitionMsvcShowIncludes(lines);
                lines.Clear();
                lines.AddRange(notIncludeLines);
            }
        }

        private static Dictionary<string, string> MakeEnvironmentVariablesForCompile(MsvcToolPathProvider toolPathProvider, IReadOnlyCollection<FilePath> includeDirectories)
        {
            return new Dictionary<string, string>()
            {
                { "PATH", toolPathProvider.VcToolPath },
                { "INCLUDE", string.Join(";", includeDirectories.Concat(toolPathProvider.IncludeDirectories).Select(x => x.PathString)) },
            };
        }

        [NactActionFunction(Version = MsvcActionVersion)]
        public static NactActionResult MsvcArchiveAction(
            INactActionContext context,
            MsvcToolChainParameters toolChainParameters,
            MsvcToolPathProvider toolPathProvider,
            FilePath outputArchiveFile,
            IReadOnlyCollection<FilePath> objectFiles,
            IReadOnlyCollection<FilePath> libraryFiles)
        {
            var helper = context.Helper;

            var flags = new List<string>();
            MsvcCommonOptions.GetLibOptions(flags, toolChainParameters);

            flags.Add(Invariant($"/OUT:\"{outputArchiveFile}\""));

            foreach (var inputFile in objectFiles.Concat(libraryFiles))
            {
                flags.Add(Invariant($"\"{inputFile}\""));
            }

            var env = new Dictionary<string, string>()
            {
                { "PATH", toolPathProvider.VcToolPath },
                { "LIB", string.Join(";", toolPathProvider.LibraryDirectories.Select(x => x.PathString)) },
            };

            var result = helper.ExecuteProgram(
                toolPathProvider.LibExePath,
                string.Join(" ", flags),
                env,
                outputModifier: null,
                warningDetector: MsvcHelper.WarningDetector,
                errorDetector: MsvcHelper.ErrorDetector);

            if (result.ExitCode != 0)
            {
                return helper.FinishAsFailure();
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction(Version = MsvcActionVersion)]
        public static NactActionResult MsvcLinkAction(
            INactActionContext context,
            MsvcToolChainParameters toolChainParameters,
            MsvcToolPathProvider toolPathProvider,
            FilePath outputProgramFile,
            FilePath outputPdbFile,
            IReadOnlyCollection<FilePath> objectFiles,
            IReadOnlyCollection<FilePath> libraryFiles,
            IReadOnlyCollection<string> systemLibraryNames,
            IReadOnlyCollection<string> linkOptions,
            bool tentativeGenerateDebugInformationOnMsvcToolchainBuild)
        {
            var helper = context.Helper;

            var flags = new List<string>();
            MsvcCommonOptions.GetLinkOptions(flags, toolChainParameters);

            flags.Add(Invariant($"/PDB:\"{outputPdbFile}\""));
            flags.Add(Invariant($"/OUT:\"{outputProgramFile}\""));

            foreach (var inputFile in objectFiles.Concat(libraryFiles))
            {
                flags.Add(Invariant($"\"{inputFile}\""));
            }

            foreach (var libraryName in systemLibraryNames)
            {
                flags.Add(Invariant($"{libraryName}.lib"));
            }

            if (!tentativeGenerateDebugInformationOnMsvcToolchainBuild)
            {
                flags.RemoveAll(x => x.StartsWith("/DEBUG"));
            }

            flags.AddRange(linkOptions);

            var env = new Dictionary<string, string>()
            {
                { "PATH", toolPathProvider.VcToolPath },
                { "LIB", string.Join(";", toolPathProvider.LibraryDirectories.Select(x => x.PathString)) },
            };

            var result = helper.ExecuteProgram(
                toolPathProvider.LinkExePath,
                string.Join(" ", flags),
                env,
                outputModifier: LinkExeOutputModifier,
                warningDetector: MsvcHelper.WarningDetector,
                errorDetector: MsvcHelper.ErrorDetector);

            if (result.ExitCode != 0)
            {
                // TODO: OutputModifier に渡すときに文字列化してあるのだから、INactActionHelper.ExecuteProgram が文字列 (lines) を返して欲しい
                if (IsTransientError(result, IsLinkExeTransientError))
                {
                    // TODO: 一過性のエラーを返すよいインターフェース
                    return helper.FinishAsFailure().AsTransientFailure();
                }
                else
                {
                    return helper.FinishAsFailure();
                }
            }

            return helper.FinishAsSuccess();
        }

        private static bool IsTransientError(ProgramExecutionResult result, Func<string, bool> isTransientErrorLine)
        {
            var lines = NactActionHelper.BytesToStringLines(result.MixedOutput, Encoding.Default, fallsbackToAscii: true);
            return lines.Any(isTransientErrorLine);
        }

        private static void ClExeOutputModifier(string sourceFileName, List<string> text)
        {
            // 1行目にソースファイル名が出力されるので除去する。
            // ただし、コマンドラインに対する警告はソースファイル名より前に出力されるので、その場合は除去しないようにする。
            if (text.Count > 0 && text[0] == sourceFileName)
            {
                text.RemoveAt(0);
            }
        }

        private static bool IsClExeTransientError(string line)
        {
            return
                // C1001 INTERNAL COMPILER ERROR, SIGLO-61548
                line.Contains("fatal error C1001")
                // コンパイラの生成した ファイルを開けません, SIGLO-36184
                || line.Contains("fatal error C1083")
                // PDB API の呼び出しに失敗しました, SIGLONTD-7908
                || line.Contains("fatal error C1090")
                // プログラム データベース '...' を更新できません。 SIGLO-82795
                || line.Contains("error C2471");
        }

        private static void LinkExeOutputModifier(List<string> lines)
        {
            lines.RemoveAll(isLineToRemove);

            bool isLineToRemove(string line)
            {
                // NOTE: /ZI で生成されたライブラリをリンクすると /EDITANDCONTINUE が自動的に付加され、抑制できない。
                //       ライブラリ側はエディットアンドコンティニューの PDB を生成しておき、必要に応じてエディットアンドコンティニューを有効にしたい。
                //       一方で、リンクはデフォルトでは /INCREMENTAL:NO を使用したい (インクリメンタルリンクはまれに失敗するため、また、常に同じ生成物が欲しいため)。
                // "warning LNK4075: /EDITANDCONTINUE は /OPT:LBR の指定によって無視されます。":
                if (line.Contains("warning LNK4075") && line.Contains("/EDITANDCONTINUE"))
                {
                    return true;
                }

                return false;
            }
        }

        public static bool IsLinkExeTransientError(string line)
        {
            return
                // SEH Exception, SIGLO-32954
                line.Contains("fatal error LNK1000")
                // 予期しない PDB エラー, SIGLONTD-7908
                || line.Contains("fatal error LNK1318");
        }
    }
}
