﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using Nintendo.Nact.Utilities.ProgramExecution;
using SigloNact.Build;
using SigloNact.Utilities;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using static Nintendo.Nact.Extensions.FormattableStringExtensions;
using static System.FormattableString;

namespace SigloNact.BuiltIns.ToolChain.GccClang
{
    [NactActionFunctionContainer]
    public static class GccClangActions
    {
        private static readonly Encoding UTF8NoBomEncoding = new UTF8Encoding(false);

        [NactActionFunction]
        public static NactActionResult GccClangCompileAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputObjectFile,
            FilePath sourceFile,
            string language,
            string precompiledHeaderMode,
            IReadOnlyCollection<string> modelOptions,
            IReadOnlyCollection<string> compileOptions,
            IReadOnlyCollection<string> preprocessorMacros,
            IReadOnlyCollection<FilePath> includeDirectories,
            FilePath precompiledHeaderSourceFile,
            FilePath precompiledHeaderObjectFile)
        {
            var languageParsed = Util.ParseEnum<CompileLanguage>(language);
            var precompiledHeaderModeParsed = Util.ParseEnum<PrecompiledHeaderMode>(precompiledHeaderMode);

            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            var flags = new List<string>();

            flags.AddRange(modelOptions);

            flags.AddRange(GccClangUtil.GetLanguageOptions(languageParsed));

            switch (precompiledHeaderModeParsed)
            {
                case PrecompiledHeaderMode.None:
                    break;
                case PrecompiledHeaderMode.Use:
                    // 拡張子を .h.gch から .h に置換
                    flags.Add("-include");
                    flags.Add(precompiledHeaderObjectFile.ChangeExtension(null).PathString);
                    break;
                case PrecompiledHeaderMode.Create:
                    // TODO: 要求はないが、いちおう、c-header もあり得る
                    // TODO: GetLanguageOptions のオプションと被っている
                    flags.Add("-x");
                    flags.Add("c++-header");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            flags.AddRange(includeDirectories.Select(x => Invariant($"-I{x.PathString}")));
            flags.AddRange(preprocessorMacros.Select(x => MakePreprocessorMacroFlag(x)));

            // 追加オプション
            flags.AddRange(compileOptions);

            // 入力ファイル、出力ファイル
            flags.Add("-c");
            flags.Add(sourceFile.PathString);
            flags.Add("-o");
            flags.Add(outputObjectFile.PathString);

            var result = ExecuteProgram(helper, commandHelper.CcPath, flags, commandHelper.EnvironmentVariables);
            if (result.ExitCode != 0)
            {
                HandleCommandFailure(commandHelper, result);
                return helper.FinishAsFailure();
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction]
        public static NactActionResult GccClangPreprocessAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputPreprocessedFile,
            FilePath sourceFile,
            string language,
            IReadOnlyCollection<string> modelOptions,
            IReadOnlyCollection<string> compileOptions,
            IReadOnlyCollection<string> preprocessorMacros,
            IReadOnlyCollection<FilePath> includeDirectories)
        {
            var languageParsed = Util.ParseEnum<CompileLanguage>(language);

            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            var flags = new List<string>();

            // FIXME: コピペ
            flags.AddRange(modelOptions);

            flags.AddRange(GccClangUtil.GetLanguageOptions(languageParsed));

            flags.AddRange(includeDirectories.Select(x => Invariant($"-I{x.PathString}")));
            flags.AddRange(preprocessorMacros.Select(x => MakePreprocessorMacroFlag(x)));

            // 追加オプション
            flags.AddRange(compileOptions);

            // プリプロセス
            flags.Add("-P");
            flags.Add("-E");

            // 入力ファイル
            flags.Add("-c");
            flags.Add(sourceFile.PathString);

            flags.Add("-o");
            flags.Add(outputPreprocessedFile.PathString);

            var result = ExecuteProgramForStandardOutput(helper, commandHelper.CcPath, flags, commandHelper.EnvironmentVariables);
            if (result.ExitCode != 0)
            {
                HandleCommandFailure(commandHelper, result);
                return helper.FinishAsFailure();
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction]
        public static NactActionResult GccClangMakeDependenciesAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputDependenciesFile,
            FilePath sourceFile,
            string language,
            IReadOnlyCollection<string> modelOptions,
            IReadOnlyCollection<string> compileOptions,
            IReadOnlyCollection<string> preprocessorMacros,
            IReadOnlyCollection<FilePath> includeDirectories)
        {
            var languageParsed = Util.ParseEnum<CompileLanguage>(language);

            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            var flags = new List<string>();

            // FIXME: コピペ
            flags.AddRange(modelOptions);

            flags.AddRange(GccClangUtil.GetLanguageOptions(languageParsed));

            flags.AddRange(includeDirectories.Select(x => Invariant($"-I{x.PathString}")));
            flags.AddRange(preprocessorMacros.Select(x => MakePreprocessorMacroFlag(x)));

            // 追加オプション
            flags.AddRange(compileOptions);

            // 警告を抑制
            flags.Add("-w");

            // dependencies を出力
            flags.Add("-M");

            // 入力ファイル
            flags.Add("-c");
            flags.Add(sourceFile.PathString);

            var result = ExecuteProgramForStandardOutput(helper, commandHelper.CcPath, flags, commandHelper.EnvironmentVariables);
            if (result.ExitCode != 0)
            {
                HandleCommandFailure(commandHelper, result);
                return helper.FinishAsFailure();
            }

            helper.WriteAllBytes(outputDependenciesFile, result.StandardOutput);

            return helper.FinishAsSuccess();
        }

        [NactActionFunction]
        public static NactActionResult GccClangArchiveAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputArchiveFile,
            IReadOnlyCollection<FilePath> objectFiles)
        {
            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            var resFile = AddExtension(outputArchiveFile, ".res");
            CreateResFile(helper, resFile, objectFiles);

            // TODO: helper でやる。DeleteFile はファイルの生成 (writeFiles への追加) とみなす。
            // DeleteFile してファイルを生成しないことは許さないということ。
            if (File.Exists(outputArchiveFile.PathString))
            {
                File.Delete(outputArchiveFile.PathString);
            }

            var flags = new List<string>();
            flags.Add("crsD");
            flags.Add(outputArchiveFile.PathString);
            flags.Add(Invariant($"@{resFile}"));

            // TORIAEZU: (SIGLO-32091) 他の ar.exe インスタンスと一時ファイル名が同じになることにより read.*.tlog に
            // 一時ファイルが含まれてしまい「常に実行されるルール」警告が出力されてしまうことの回避。
            // ArchiveRule の入力はビルドルールで正しく管理されている前提で、TrackFileAccess を無効にする。

            var (result, outputText) = helper.DangerousExecuteProgram(
                commandHelper.ArPath,
                string.Join(" ", flags),
                commandHelper.EnvironmentVariables.ToDictionary(x => x.Key, x => x.Value),
                flags: NactExecuteProgramFlags.None);

            helper.AddReadFile(commandHelper.ArPath);
            helper.AddReadFiles(objectFiles);
            helper.AddWriteFile(outputArchiveFile);
            helper.AddWriteFile(resFile);
            helper.HandleProgramExecutionResult(
                result,
                outputText,
                commandHelper.ArPath,
                outputModifier: GccClangUtil.OutputModifier,
                errorDetector: GccClangUtil.ErrorDetector,
                warningDetector: GccClangUtil.WarningDetector);

            if (result.ExitCode != 0)
            {
                HandleCommandFailure(commandHelper, result);
                return helper.FinishAsFailure();
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction]
        public static NactActionResult GccClangLinkAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputProgramFile,
            FilePath outputMapFile,
            FilePath outputDasmFile,
            IReadOnlyCollection<IReadOnlyCollection<object>> linkGroups,
            IReadOnlyCollection<string> modelOptions,
            IReadOnlyCollection<string> linkOptions,
            FilePath linkScriptFile,
            FilePath versionScriptFile,
            string linkerKind,
            string NXDebugLink,
            string buildIdStyle,    // none,sha1,...
            bool linkWholeArchive,
            bool generateMap,
            bool generateDasm,
            bool stripDebug)
        {
            var linkerKindParsed = Util.ParseEnum<LinkerKind>(linkerKind);

            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            var resFile = AddExtension(outputProgramFile, ".res");

            CreateLinkGroupsResFile(helper, resFile, linkGroups);

            var flags = new List<string>();

            flags.AddRange(modelOptions);

            // その他の引数に対応するオプション
            if (generateMap)
            {
                flags.Add(Invariant($"-Wl,-Map={outputMapFile}"));
                // なんでここに -S があるんだろう？ 安達さん？
                if (stripDebug)
                {
                    flags.Add("-Wl,-S");
                }
            }

            {
                var linkerName = GccClangUtil.GetLinkerNameForUseLd(toolChainSpecifier.ToolChainKind, linkerKindParsed);
                flags.Add(Current($"-fuse-ld={linkerName}"));
            }

            if (toolChainSpecifier.ToolChainKind == ToolChainKind.Clang)
            {
                if (NXDebugLink != null)
                {
                    flags.Add(Invariant($"-Wl,--add-nx-debuglink=\"{NXDebugLink}\""));
                }
                else
                {
                    flags.Add(Invariant($"-Wl,--no-add-nx-debuglink"));
                }
            }

            flags.Add(Invariant($"-Wl,--build-id={buildIdStyle}"));

            // 追加オプション
            flags.AddRange(linkOptions);

            // 出力ファイル、入力ファイル
            if (linkWholeArchive)
            {
                // NOTE: LD の機能としては、本当はアーカイブごとに --whole-archive, --no-whole-archive を指定することが可能
                flags.Add("-Wl,--whole-archive");
            }

            flags.Add("-o");
            flags.Add(outputProgramFile.PathString);

            // 入力ファイル
            flags.Add(Invariant($"@{resFile}"));

            // リンカスクリプトにリンクグループを書いており、かつリンカスクリプトがオブジェクトファイルより後に指定されることに依存しているリンクがある
            // リンカスクリプトは入力ファイルの最後として指定することにする。
            if (linkScriptFile != null)
            {
                flags.Add(Invariant($"-Wl,--script={linkScriptFile.PathString}"));
            }

            if (versionScriptFile != null)
            {
                flags.Add(Invariant($"-Wl,--version-script={versionScriptFile.PathString}"));
            }

            var result = ExecuteProgram(helper, commandHelper.LdPath, flags, commandHelper.EnvironmentVariables);
            if (result.ExitCode != 0)
            {
                HandleCommandFailure(commandHelper, result);
                return helper.FinishAsFailure();
            }

            if (generateDasm)
            {
                var objdumpResult = ExecuteProgramForStandardOutput(
                    helper, commandHelper.ObjdumpPath, new string[] { "-d", outputProgramFile.PathString }, commandHelper.EnvironmentVariables);

                helper.WriteAllBytes(outputDasmFile, objdumpResult.StandardOutput);

                if (objdumpResult.ExitCode != 0)
                {
                    HandleCommandFailure(commandHelper, objdumpResult);
                    return helper.FinishAsFailure();
                }
            }
            else if (outputDasmFile != null)
            {
                // TORIAEZU: 過去に生成した dasm ファイルが存在していれば、古いものを参照しないように書きつぶしておく
                // TODO: 生成しなくなったファイルを削除するのはエンジンの責任
                if (File.Exists(outputDasmFile.PathString))
                {
                    helper.WriteAllBytes(outputDasmFile, Array.Empty<byte>());
                }
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction]
        public static NactActionResult GccClangElf2BinAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputBinaryFile,
            FilePath objectFile,
            IReadOnlyCollection<string> RemoveSections)
        {
            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            var flags = new List<string>();
            flags.Add("-O");
            flags.Add("binary");
            foreach (var removeSection in RemoveSections)
            {
                flags.Add("-R");
                flags.Add(removeSection);
            }
            flags.Add(objectFile.PathString);
            flags.Add(outputBinaryFile.PathString);

            var result = ExecuteProgram(helper, commandHelper.ObjcopyPath, flags, commandHelper.EnvironmentVariables);
            if (result.ExitCode != 0)
            {
                HandleCommandFailure(commandHelper, result);
                return helper.FinishAsFailure();
            }

            return helper.FinishAsSuccess();
        }

        [NactActionFunction]
        public static NactActionResult GccClangRemoveDebugInformationAction(
            INactActionContext context,
            GccClangToolChainSpecifier toolChainSpecifier,
            FilePath outputFile,
            FilePath inputFile,
            IReadOnlyCollection<string> targetSections,
            bool keepDebugFrame)
        {
            var helper = context.Helper;
            var commandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);

            context.CancellationToken.ThrowIfCancellationRequested();

            using (var temporaryDir = new TemporaryDirectory(context.TemporaryDirectory))
            {
                var flags = new List<string>();
                if (keepDebugFrame)
                {
                    var debugFrameFile = temporaryDir.Path.Combine("debug_frame");
                    var strippedFile = temporaryDir.Path.Combine("stripped");

                    {
                        flags.Clear();
                        flags.Add("--dump-section");
                        flags.Add(Invariant($".debug_frame=\"{debugFrameFile}\""));
                        flags.Add(inputFile.PathString);
                        // .debug_frame セクションをダンプしたいだけなのだが、
                        // 出力ファイルを指定しないと objcopy が入力ファイルを上書きするので、ダミーの出力ファイルを指定（SIGLO-17382）
                        flags.Add("NUL");
                        var result = ExecuteProgram(helper, commandHelper.ObjcopyPath, flags, commandHelper.EnvironmentVariables);
                        if (result.ExitCode != 0)
                        {
                            HandleCommandFailure(commandHelper, result);
                            return helper.FinishAsFailure();
                        }
                    }

                    {
                        flags.Clear();
                        flags.Add("--strip-debug");
                        flags.Add("--strip-unneeded");
                        flags.AddRange(targetSections.Select(x => Invariant($"--remove-section={x}")));
                        flags.Add("-o");
                        flags.Add(strippedFile.PathString);
                        flags.Add(inputFile.PathString);
                        var result = ExecuteProgram(helper, commandHelper.StripPath, flags, commandHelper.EnvironmentVariables);
                        if (result.ExitCode != 0)
                        {
                            HandleCommandFailure(commandHelper, result);
                            return helper.FinishAsFailure();
                        }
                    }

                    {
                        flags.Clear();
                        flags.Add("--add-section");
                        flags.Add(Invariant($".debug_frame=\"{debugFrameFile}\""));
                        flags.Add(strippedFile.PathString);
                        flags.Add(outputFile.PathString);

                        var result = ExecuteProgram(helper, commandHelper.ObjcopyPath, flags, commandHelper.EnvironmentVariables);
                        if (result.ExitCode != 0)
                        {
                            HandleCommandFailure(commandHelper, result);
                            return helper.FinishAsFailure();
                        }
                    }
                }
                else
                {
                    {
                        flags.Clear();
                        flags.Add("--strip-debug");
                        flags.Add("--strip-unneeded");
                        flags.AddRange(targetSections.Select(x => Invariant($"--remove-section={x}")));
                        flags.Add("-o");
                        flags.Add(outputFile.PathString);
                        flags.Add(inputFile.PathString);
                        var result = ExecuteProgram(helper, commandHelper.StripPath, flags, commandHelper.EnvironmentVariables);
                        if (result.ExitCode != 0)
                        {
                            HandleCommandFailure(commandHelper, result);
                            return helper.FinishAsFailure();
                        }
                    }
                }
            }

            return helper.FinishAsSuccess();
        }

        private static ProgramExecutionResult ExecuteProgram(
            INactActionHelper helper,
            FilePath executablePath,
            IReadOnlyCollection<string> flags,
            IReadOnlyDictionary<string, string> environmentVariables)
        {
            return helper.ExecuteProgram(
                executablePath,
                string.Join(" ", flags),
                environmentVariables.ToDictionary(x => x.Key, x => x.Value),
                outputModifier: GccClangUtil.OutputModifier,
                errorDetector: GccClangUtil.ErrorDetector,
                warningDetector: GccClangUtil.WarningDetector);
        }

        private static ProgramExecutionResult ExecuteProgramForStandardOutput(
            INactActionHelper helper,
            FilePath executablePath,
            IReadOnlyCollection<string> flags,
            IReadOnlyDictionary<string, string> environmentVariables)
        {
            return helper.ExecuteProgramForStandardOutput(
                executablePath,
                string.Join(" ", flags),
                environmentVariables.ToDictionary(x => x.Key, x => x.Value),
                outputModifier: GccClangUtil.OutputModifier,
                errorDetector: GccClangUtil.ErrorDetector,
                warningDetector: GccClangUtil.WarningDetector);
        }


        private static FilePath AddExtension(FilePath path, string extension) =>
            path.Parent.Combine(path.Leaf.OriginalValue + extension);

        private static void CreateResFile(
            INactActionHelper helper,
            FilePath resFilePath,
            IEnumerable<FilePath> filePaths)
        {
            var resFileText = new StringBuilder();
            foreach (var x in filePaths)
            {
                resFileText.AppendFormat("\"{0}\"\n", x.PathString.Replace('\\', '/'));
            }
            helper.WriteAllText(resFilePath, resFileText.ToString(), UTF8NoBomEncoding);
        }

        private static void CreateLinkGroupsResFile(INactActionHelper helper, FilePath resFilePath, IReadOnlyCollection<IReadOnlyCollection<object>> linkGroups)
        {
            // NOTE: オブジェクト・ライブラリの含め方を「完全に制御できる」ようにするには、例えば、順番に指定するのだ、というふうにする。
            // IReadOnlyCollection<IReadOnlyCollection<object>> linkGroups =
            // [
            //    [f"crtBegin.o"],
            //    [f"hoge.o", f"fuga.o", f"piyo.a", SystemLibrary("c")],
            //    [f"hoge.nrs"],
            //    [f"fuga.nrs"],
            //    [f"hoge.nss", "fuga.nss"],
            //    [f"crtEnd.o"],
            // ]

            using (var sw = new StreamWriter(helper.CreateFile(resFilePath)))
            {
                foreach (var group in linkGroups)
                {
                    if (group.Count == 0)
                    {
                        continue;
                    }

                    bool singleFile = group.Count == 1;

                    if (!singleFile)
                    {
                        sw.WriteLine("-Wl,--start-group");
                    }

                    foreach (var x in group)
                    {
                        switch (x)
                        {
                            case FilePath file:
                                {
                                    var normalizedPath = file.PathString.Replace('\\', '/');
                                    sw.WriteLine(Invariant($"\"{normalizedPath}\""));
                                }
                                break;
                            case SystemLibrary systemLibrary:
                                sw.WriteLine(Invariant($"-l{systemLibrary.Name}"));
                                break;
                            default:
                                throw new ArgumentException($"LinkGroup contains invalid value {x} of type {x.GetType().FullName}.");
                        }
                    }

                    if (!singleFile)
                    {
                        sw.WriteLine("-Wl,--end-group");
                    }
                }
            }
        }

        public static string MakePreprocessorMacroFlag(string macro) =>
            string.Format("-D \"{0}\"", InternalUtilities.EscapeDoubleQuotesInPreprocessorMacro(macro));

        private static void HandleCommandFailure(IGccClangCommandHelper commandHelper, ProgramExecutionResult result)
        {
            var errorLines = NactActionHelper.BytesToStringLines(result.StandardError, Encoding.Default, fallsbackToAscii: true);
            PublishBugReportFilesToTeamCity(commandHelper, errorLines);
        }

        private static void PublishBugReportFilesToTeamCity(IGccClangCommandHelper commandHelper, IReadOnlyList<string> errorText)
        {
            foreach (var path in commandHelper.GetBugReportFilesFromOutput(errorText))
            {
                Log.Message("##teamcity[publishArtifacts '{0}']", path);
            }
        }
    }
}
