﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;

namespace SigloNact.BuiltIns.FsTest
{
    [NactFunctionContainer]
    [NactAction]
    public class CreateFsTestResource : INactAction
    {
        private readonly string m_MethodName;
        private readonly FilePath m_TargetFilePath;
        private readonly string m_Size;
        private readonly string m_Num;

        public ImmutableArray<object> NactObjectCreationArguments { get; }

        [NactFunction]
        public static IEnumerable<string> GetNamesCreatedByFsTestResourceCreator(string methodName, string name, string size, string num)
        {
            switch (methodName)
            {
                case "CreateZeroDataFile":
                case "CreateFileNameSeedRandomDataFile":
                case "CreateFilledDataFile":
                    yield return name;
                    break;

                case "CreateFilesInOrder":
                    {
                        var fileNum = Convert.ToInt64(num, 10);
                        var dirPath = Path.GetDirectoryName(name);
                        for (uint cnt = 0; cnt < fileNum; cnt++)
                        {
                            string path = dirPath + "\\" + cnt.ToString("0000") + ".file";
                            yield return path;
                        }
                    }
                    break;

                case "CreateFlatDirectoriesAndFile":
                    {
                        var dirPath = Path.GetDirectoryName(name);
                        var dirNum = Convert.ToInt64(num, 10);

                        yield return name;

                        for (uint cnt = 0; cnt < dirNum; cnt++)
                        {
                            string createDirPath = dirPath + "\\" + cnt.ToString("0000");
                            string createFilePath = createDirPath + "\\test.file";
                            yield return createFilePath;
                        }

                    }
                    break;

                case "CreateNestDirectoriesAndFile":
                    {
                        var dirPath = Path.GetDirectoryName(name);
                        string workPath = dirPath;
                        var dirNum = Convert.ToInt64(num, 10);

                        yield return name;

                        for (uint cnt = 0; cnt < dirNum; cnt++)
                        {
                            uint n = cnt % 10;
                            workPath = workPath + "\\" + n.ToString("0");
                            string createFilePath = workPath + "\\test.file";
                            yield return createFilePath;
                        }
                    }
                    break;

                default:
                    throw new Exception($"Unknown MethodName = {methodName}");
            }
        }

        [NactFunction]
        public IEnumerable<object> SkipIfNotModified => Enumerable.Empty<object>();

        [NactObjectCreator]
        public CreateFsTestResource(string methodName, FilePath targetFilePath, string size, string num = null)
        {
            m_MethodName = methodName;
            m_TargetFilePath = targetFilePath;
            m_Size = size;
            m_Num = num;

            NactObjectCreationArguments = ImmutableArray.Create<object>(methodName, targetFilePath, size, num);
        }

        public NactActionResult Execute(INactActionContext context)
        {
            switch (m_MethodName)
            {
                case "CreateZeroDataFile":
                    CreateZeroDataFile(m_TargetFilePath.PathString, m_Size);
                    break;
                case "CreateFileNameSeedRandomDataFile":
                    CreateFileNameSeedRandomDataFile(m_TargetFilePath.PathString, m_Size);
                    break;
                case "CreateFilledDataFile":
                    CreateFilledDataFile(m_TargetFilePath.PathString, m_Size);
                    break;
                case "CreateFilesInOrder":
                    CreateFilesInOrder(m_TargetFilePath.PathString, m_Size, m_Num);
                    break;
                case "CreateFlatDirectoriesAndFile":
                    CreateFlatDirectoriesAndFile(m_TargetFilePath.PathString, m_Size, m_Num);
                    break;
                case "CreateNestDirectoriesAndFile":
                    CreateNestDirectoriesAndFile(m_TargetFilePath.PathString, m_Size, m_Num);
                    break;
                default:
                    throw new Exception($"Unknown MethodName = {m_MethodName}");
            }
            return NactActionResult.CreateSuccess(
                Array.Empty<FilePath>(),
                new[] { m_TargetFilePath });
        }

        private class XorShift
        {
            private uint m_X;
            private uint m_Y;
            private uint m_Z;
            private uint m_W;

            public XorShift(ulong seed)
            {
                m_X = 123456789 ^ (uint)((seed >> 32) & 0xFFFFFFFF);
                m_Y = 362436069 ^ (uint)(seed & 0xFFFFFFFF);
                m_Z = 521288629;
                m_W = 88675123;
            }

            private static ulong ConvertStrToUInt64(string str)
            {
                var strBytes = Encoding.ASCII.GetBytes(str);
                Array.Resize(ref strBytes, 8);
                return BitConverter.ToUInt64(strBytes, 0);
            }

            public XorShift(string strSeed) : this(XorShift.ConvertStrToUInt64(strSeed))
            {
            }

            public uint NextUInt32()
            {
                uint t = m_X ^ (m_X << 11);
                m_X = m_Y;
                m_Y = m_Z;
                m_Z = m_W;
                return (m_W = (m_W ^ (m_W >> 19)) ^ (t ^ (t >> 8)));
            }

            public byte[] NextByte(int length)
            {
                var data = new byte[length];
                using (var stream = new MemoryStream(data))
                {
                    while (stream.Position != stream.Length)
                    {
                        var random = BitConverter.GetBytes(NextUInt32());
                        var size = (int)Math.Min(stream.Length - stream.Position, random.Length);
                        stream.Write(random, 0, size);
                    }
                }
                return data;
            }
        }

        private static void CreateZeroDataFile(string filePath, string fileSizeStr)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(dirPath))
            {
                Directory.CreateDirectory(dirPath);
            }
            var fileSize = Convert.ToInt64(fileSizeStr, 16);
            using (var stream = new FileStream(filePath, FileMode.Create, FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
            {
                stream.SetLength(fileSize);
            }
        }

        private static void CreateFileNameSeedRandomDataFile(string filePath, string fileSizeStr)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(Path.GetFullPath(dirPath)))
            {
                Directory.CreateDirectory(dirPath);
            }
            var fileSize = Convert.ToInt64(fileSizeStr, 16);
            const int BufferSize = 1024 * 1024;
            var rng = new XorShift(Path.GetFileName(filePath));
            using (var stream = new FileStream(filePath, FileMode.Create, FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
            {
                while (fileSize - stream.Position > 0)
                {
                    int size = (int)Math.Min((long)BufferSize, fileSize - stream.Position);
                    var data = rng.NextByte(size);
                    stream.Write(data, 0, data.Length);
                }
            }
        }

        private static void CreateFilledData(string filePath, long fileSize)
        {
            using (var stream = new FileStream(filePath, FileMode.Create, FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
            {
                uint cnt = 0;

                while (fileSize - stream.Position > 0)
                {
                    byte[] data = new byte[1] { (byte)cnt };
                    stream.Write(data, 0, 1);
                    cnt++;
                }
            }
        }

        private static void CreateFilledDataFile(string filePath, string fileSizeStr)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(Path.GetFullPath(dirPath)))
            {
                Directory.CreateDirectory(dirPath);
            }
            var fileSize = Convert.ToInt64(fileSizeStr, 16);

            CreateFilledData(filePath, fileSize);
        }

        private static void CreateFilesInOrder(string filePath, string fileSizeStr, string fileNumStr)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(Path.GetFullPath(dirPath)))
            {
                Directory.CreateDirectory(dirPath);
            }

            var fileSize = Convert.ToInt64(fileSizeStr, 16);
            var fileNum = Convert.ToInt64(fileNumStr, 10);

            for (uint cnt = 0; cnt < fileNum; cnt++)
            {
                string path = dirPath + "\\" + cnt.ToString("0000") + ".file";
                CreateFilledData(path, fileSize);
            }
        }

        private static void CreateFlatDirectoriesAndFile(string filePath, string fileSizeStr, string dirNumStr)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(Path.GetFullPath(dirPath)))
            {
                Directory.CreateDirectory(dirPath);
            }

            var fileSize = Convert.ToInt64(fileSizeStr, 16);
            var dirNum = Convert.ToInt64(dirNumStr, 10);

            CreateFilledData(filePath, fileSize);

            for (uint cnt = 0; cnt < dirNum; cnt++)
            {
                string createDirPath = dirPath + "\\" + cnt.ToString("0000");
                Directory.CreateDirectory(createDirPath);

                string createFilePath = createDirPath + "\\test.file";
                CreateFilledData(createFilePath, fileSize);
            }
        }

        private static void CreateNestDirectoriesAndFile(string filePath, string fileSizeStr, string dirNumStr)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(Path.GetFullPath(dirPath)))
            {
                Directory.CreateDirectory(dirPath);
            }

            string workPath = dirPath;
            var fileSize = Convert.ToInt64(fileSizeStr, 16);
            var dirNum = Convert.ToInt64(dirNumStr, 10);

            CreateFilledData(filePath, fileSize);

            for (uint cnt = 0; cnt < dirNum; cnt++)
            {
                uint n = cnt % 10;
                workPath = workPath + "\\" + n.ToString("0");
                Directory.CreateDirectory(workPath);

                string createFilePath = workPath + "\\test.file";
                CreateFilledData(createFilePath, fileSize);
            }
        }
    }
}
