﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.IO;
using System.Linq;
using System.Threading;

namespace SigloNact.BuiltIns.FsTest
{
    [NactAction]
    public class CreateFsTestIntegrationNcaResource : INactAction
    {
        private readonly FilePath m_TargetFilePath;
        private readonly string m_Generation;

        [NactFunction]
        public IEnumerable<object> SkipIfNotModified => Array.Empty<object>();

        public ImmutableArray<object> NactObjectCreationArguments { get; }

        [NactObjectCreator]
        public CreateFsTestIntegrationNcaResource(FilePath targetFilePath, string generation)
        {
            m_TargetFilePath = targetFilePath;
            m_Generation = generation;
            NactObjectCreationArguments = ImmutableArray.Create<object>(targetFilePath, generation);
        }

        public NactActionResult Execute(INactActionContext context)
        {
            CreateDiff(m_TargetFilePath.PathString, m_Generation);

            return NactActionResult.CreateSuccess(
                Array.Empty<FilePath>(),
                new[] { m_TargetFilePath });
        }

        const int FileVariation = 100;
        const int ReplaceVariation = 1024;
        const int PositionStrideMin = 4 * 1024;
        const int PositionStrideMax = 128 * 1024;

        private static void CreateDiff(string mountFilePath, string generationStr)
        {
            var inputRootDir = Path.GetDirectoryName(mountFilePath);

            var now = DateTime.Now;
            var seed = now.Year;
            seed = seed * 100 + now.Month;
            seed = seed * 100 + now.Day;
            seed = seed * 100 + now.Hour;
            var random = new Random(seed);

            var buffer = new byte[PositionStrideMax];
            var generation = int.Parse(generationStr);

            for (int i = 0; i < generation; ++i)
            {
                foreach (var inputFilePath in Directory.EnumerateFiles(inputRootDir, "*", SearchOption.AllDirectories))
                {
                    if (inputFilePath == mountFilePath)
                    {
                        continue;
                    }

                    var inputDir = Path.GetDirectoryName(inputFilePath);

                    if (random.Next(FileVariation) == 0)
                    {
                        var copyFilePath = inputFilePath + ".tmp";

                        File.Copy(inputFilePath, copyFilePath, true);

                        using (var inStream = new FileStream(copyFilePath, FileMode.Open))
                        using (var reader = new BinaryReader(inStream))
                        using (var outStream = new FileStream(inputFilePath, FileMode.Create))
                        using (var writer = new BinaryWriter(outStream))
                        {
                            while (inStream.Position < inStream.Length)
                            {
                                int stride = random.Next(PositionStrideMin, PositionStrideMax);

                                stride = reader.Read(buffer, 0, stride);
                                writer.Write(buffer, 0, stride);

                                // 元データを超えないように、かつ、末尾には到達するように。
                                var size = (int)Math.Min(inStream.Length - inStream.Position, random.Next(8, ReplaceVariation));

                                // 元データは単にシーク
                                inStream.Seek(size, SeekOrigin.Current);

                                // 新データはランダムデータを書き込み
                                var randomBytes = new byte[size];
                                random.NextBytes(randomBytes);
                                writer.Write(randomBytes);
                            }
                        }

                        File.Delete(copyFilePath);
                    }
                }
            }

            using (var listFile = new StreamWriter(mountFilePath, false))
            {
                listFile.NewLine = "\n";
                listFile.WriteLine(string.Format("{0}", inputRootDir));
            }
        }
    }
}
