﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

namespace SigloNact.BuiltIns.DeviceTree
{
    [NactActionFunctionContainer]
    public static class CompileDeviceTreeContainer
    {
        /// <summary>
        ///
        /// </summary>
        /// <param name="context"></param>
        /// <param name="dtcPath">Device Tree Compiler の実行ファイルのパス</param>
        /// <param name="sourcePath">ソースファイルのパス</param>
        /// <param name="includeSearchPathList">include 検索対象のパスのリスト</param>
        /// <param name="additionalCompileOptions">コンパイラに渡す追加オプション</param>
        /// <param name="outputPath">出力先ファイルのパス</param>
        /// <param name="checkLocationForReadFiles">true 時、処理実行時に読み込まれたファイルが許可ディレクトリ下にすべて含まれているかチェックする</param>
        /// <param name="targetDirectoryToCheckLocation">チェック対象のルートディレクトリ。この外のファイルはチェック対象にしない。通常は Siglo のルートディレクトリを指定</param>
        /// <param name="allowedDirectoriesForReadFiles">許可ディレクトリリスト</param>
        /// <param name="exemptedDirectoriesForReadFiles">allowedDirectoriesForReadFiles と同じ働きだが、エラーメッセージで「許可ディレクトリリスト」として表示しない</param>
        /// <returns></returns>
        [NactActionFunction]
        public static NactActionResult CompilePreprocessedDeviceTree(
            INactActionContext context,
            FilePath dtcPath,
            FilePath sourcePath,
            IEnumerable<FilePath> includeSearchPathList,
            IEnumerable<string> additionalCompileOptions,
            FilePath outputPath,
            bool checkLocationForReadFiles,
            FilePath targetDirectoryToCheckLocation,
            IEnumerable<FilePath> allowedDirectoriesForReadFiles,
            IEnumerable<FilePath> exemptedDirectoriesForReadFiles)
        {
            var helper = context.Helper;

            var args = string.Join(" ",
                "-I", "dts",
                "-O", "dtb",
                string.Join(" ", includeSearchPathList.Select(path => $"-i{path.PathString}")),
                "-o", outputPath.PathString,
                string.Join(" ", additionalCompileOptions),
                sourcePath.PathString);

            var result = helper.ExecuteProgram(dtcPath, args);

            if (!result.IsSuccessful)
            {
                return helper.FinishAsFailure();
            }

            if (checkLocationForReadFiles)
            {
                if (!CheckLocationOfReadFiles(helper, result.TrackingResult.ReadFiles, targetDirectoryToCheckLocation, allowedDirectoriesForReadFiles, exemptedDirectoriesForReadFiles))
                {
                    return helper.FinishAsFailure();
                }
            }

            return helper.FinishAsSuccess();
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="context"></param>
        /// <param name="preprocessorPath">C プリプロセッサのパス</param>
        /// <param name="sourcePath">ソースファイルのパス</param>
        /// <param name="includeSearchPathList">include 検索対象のパスのリスト</param>
        /// <param name="additionalCompileOptions">コンパイラに渡す追加オプション</param>
        /// <param name="outputPath">出力先ファイルのパス</param>
        /// <param name="checkLocationForReadFiles">true 時、処理実行時に読み込まれたファイルが許可ディレクトリ下にすべて含まれているかチェックする</param>
        /// <param name="targetDirectoryToCheckLocation">チェック対象のルートディレクトリ。この外のファイルはチェック対象にしない。通常は Siglo のルートディレクトリを指定</param>
        /// <param name="allowedDirectoriesForReadFiles">許可ディレクトリリスト</param>
        /// <param name="exemptedDirectoriesForReadFiles">allowedDirectoriesForReadFiles と同じ働きだが、エラーメッセージで「許可ディレクトリリスト」として表示しない</param>
        /// <returns></returns>
        [NactActionFunction]
        public static NactActionResult PreProcessCDirectivesForDeviceTree(
            INactActionContext context,
            FilePath preprocessorPath,
            FilePath sourcePath,
            IEnumerable<FilePath> includeSearchPathList,
            IEnumerable<string> additionalCompileOptions,
            FilePath outputPath,
            bool checkLocationForReadFiles,
            FilePath targetDirectoryToCheckLocation,
            IEnumerable<FilePath> allowedDirectoriesForReadFiles,
            IEnumerable<FilePath> exemptedDirectoriesForReadFiles)
        {
            var helper = context.Helper;

            var args = string.Join(" ",
                "-E", // nothing is done except preprocessing
                "-P", // Inhibit generation of linemarkers in the output from the preprocessor
                "-nostdinc", // 標準ライブラリのヘッダを探索対象にせず、-I で明示的に指定したところしか見ない
                "-undef", // Do not predefine any system-specific macros. The standard predefined macros remain defined.
                "-x assembler-with-cpp", // Refrain compiler from complaining unknown directive (dts uses own definition e.g. #address-cells)
                "-D__DTS__", // Following de-facto standard custom in Linux
                string.Join(" ", includeSearchPathList.Select(path => $"-I{path.PathString}")),
                "-o", outputPath.PathString,
                string.Join(" ", additionalCompileOptions),
                sourcePath.PathString);

            var result = helper.ExecuteProgram(preprocessorPath, args);

            if (!result.IsSuccessful)
            {
                return helper.FinishAsFailure();
            }

            if (checkLocationForReadFiles)
            {
                if (!CheckLocationOfReadFiles(helper, result.TrackingResult.ReadFiles, targetDirectoryToCheckLocation, allowedDirectoriesForReadFiles, exemptedDirectoriesForReadFiles))
                {
                    return helper.FinishAsFailure();
                }
            }

            return helper.FinishAsSuccess();
        }

        private static bool CheckLocationOfReadFiles(
            IExecutionSideEffectWriter writer,
            IEnumerable<FilePath> readFiles,
            FilePath targetDirectoryToCheckLocation,
            IEnumerable<FilePath> allowedDirectoriesForReadFiles,
            IEnumerable<FilePath> exemptedDirectoriesForReadFiles)
        {
            // targetDirectoryToCheckLocation （通常は Siglo のルートディレクトリ）外のファイルはチェック対象にしない
            // ReadFiles には Windows のシステム dll なども含まれてくるため、これを除外するのが目的
            var dependentFiles = readFiles
                .Distinct()
                .Where(file => targetDirectoryToCheckLocation.IsSameOrAncestorOf(file))
                .ToList();
            var unallowedFiles = Utilities.ConvertShowIncludesUtil.GetFilteredPaths(dependentFiles, allowedDirectoriesForReadFiles.Concat(exemptedDirectoriesForReadFiles));
            if (unallowedFiles.Any())
            {
                var sb = new StringBuilder();
                sb.AppendFormat("Target DeviceTree source includes file(s) that may not be allowed for product use.\n");
                sb.AppendFormat("Specify \"ForDevelopmentOnly = true\" in the rule to compile for development purpose only.\n");
                sb.AppendFormat("Following file(s) is not under any allowed path:\n");
                foreach (var file in unallowedFiles)
                {
                    sb.AppendFormat($"  {file.PathString}\n");
                }
                sb.AppendFormat("Allowed path (safe for product use) list:\n");
                foreach (var allowedPath in allowedDirectoriesForReadFiles)
                {
                    sb.AppendFormat($"  {allowedPath.PathString}\n");
                }
                writer.Out.Write(sb.ToString());
                writer.SetErrorSummary("Dependent file location check failed.");
                return false;
            }
            return true;
        }
    }
}
