﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace SigloNact.BuiltIns.BuildSystem
{
    public class ProgramExecutionAggregator : TargetRuleAggregator<string[][], ProgramExecutionAggregator.TypedCommandLineAndDependency, string>
    {
        public override TypedCommandLineAndDependency Translate(string[][] input)
        {
            return new TypedCommandLineAndDependency(input);
        }

        public override string GetGroupKey(TypedCommandLineAndDependency data)
        {
            return data.CommandLine;
        }

        public override void Validate(TypedCommandLineAndDependency[] group)
        {
            TypedCommandLineAndDependency first = null;

            foreach (var other in group)
            {
                if (first == null)
                {
                    first = other;
                }
                else
                {
                    if (!other.HasEqualOutputsWith(first))
                    {
                        throw new TargetRuleAggregatorException(string.Format(
                            "Although two command line strings are identical, different output files are specified.\n" +
                            "Command Line = {0}\n" +
                            "Outputs1     = [{1}]\n" +
                            "Outputs2     = [{2}]\n",
                            other.CommandLine, string.Join(", ", first.Outputs), string.Join(", ", other.Outputs)
                        ));
                    }
                    if (!other.HasEqualBuiltSourcesWith(first))
                    {
                        throw new TargetRuleAggregatorException(string.Format(
                            "Although two command line strings are identical, different source files are specified.\n" +
                            "Command Line  = {0}\n" +
                            "BuiltSources1 = [{1}]\n" +
                            "BuiltSources2 = [{2}]\n",
                            other.CommandLine, string.Join(", ", first.BuiltSources), string.Join(", ", other.BuiltSources)
                        ));
                    }
                    if (!other.HasEqualRuntimeSourcesWith(first))
                    {
                        throw new TargetRuleAggregatorException(string.Format(
                            "Although two command line strings are identical, different source files are specified.\n" +
                            "Command Line  = {0}\n" +
                            "RuntimeSources1 = [{1}]\n" +
                            "RuntimeSources2 = [{2}]\n",
                            other.CommandLine, string.Join(", ", first.RuntimeSources), string.Join(", ", other.RuntimeSources)
                        ));
                    }
                }
            }
        }

        public override object Translate(TypedCommandLineAndDependency output)
        {
            return new object[] { output.Program, output.Arguments, output.Outputs, output.BuiltSources, output.RuntimeSources };
        }

        public class TypedCommandLineAndDependency
        {
            public string Program { get; private set; }
            public IEnumerable<string> Arguments { get; private set; }
            public IEnumerable<string> Outputs { get; private set; }
            public IEnumerable<string> BuiltSources { get; private set; }
            public IEnumerable<string> RuntimeSources { get; private set; }

            public string CommandLine { get; private set; }

            public TypedCommandLineAndDependency(string[][] input)
            {
                if (input.Length != 5)
                {
                    throw new TargetRuleAggregatorException("list length must be 5: Program, Arguments, Outputs, BuiltSources, and RuntimeSources");
                }

                Program = input[0][0];
                if (input[0].Length > 1)
                {
                    throw new TargetRuleAggregatorException("Program must be a single file path.");
                }

                Arguments = input[1];
                Outputs = input[2];
                BuiltSources = input[3];
                RuntimeSources = input[4];

                CommandLine = Program + " " + string.Join(" ", Arguments);
            }

            public bool HasEqualOutputsWith(TypedCommandLineAndDependency tcld)
            {
                return Outputs.OrderBy(x => x).SequenceEqual(tcld.Outputs.OrderBy(x => x));
            }

            public bool HasEqualBuiltSourcesWith(TypedCommandLineAndDependency tcld)
            {
                return BuiltSources.OrderBy(x => x).SequenceEqual(tcld.BuiltSources.OrderBy(x => x));
            }

            public bool HasEqualRuntimeSourcesWith(TypedCommandLineAndDependency tcld)
            {
                return RuntimeSources.OrderBy(x => x).SequenceEqual(tcld.RuntimeSources.OrderBy(x => x));
            }
        }
    }
}
