﻿using Nintendo.Nact.FileSystem;
using Nintendo.Nact.BuiltIn;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace SigloNact.BuiltIns.BuildSystem
{
    [NactFunctionContainer]
    public static class ConvertToSubstitutedPathOrRelativePathContainer
    {
        [NactFunction]
        public static string ConvertToSubstitutedPathOrRelativePath(
            FilePath path,
            FilePath baseDirectory,
            IReadOnlyDictionary<string, IReadOnlyCollection<string>> pathProperties)
        {
            if (PathUtility.IncludesMsbuildProperty(path.PathString))
            {
                return path.PathString;
            }

            var substitutionPaths = pathProperties.ToDictionary(
                x => string.Format("$({0})", x.Key),
                x => x.Value.Single()); // 評価済み ConfigurableValue の値はリストになる

            var pathUtility = new PathUtility(baseDirectory.PathString, substitutionPaths);

            string p;
            var substituted = pathUtility.ConvertToSubstitutedPath(pathUtility.ConvertToAbsolutePath(path.PathString), out p);
            if (substituted)
            {
                return p;
            }
            else
            {
                return pathUtility.ConvertToRelativePath(path.PathString);
            }
        }

        #region MakeVisualStudioProject から実装をコピペ
        private class PathUtility
        {
            private static readonly Regex s_MsbuildPropertyRegex = new Regex(@"\$\([a-zA-Z_][0-9a-zA-Z_]*\)", RegexOptions.Compiled);

            public string BaseDirectory { get; set; }
            public IDictionary<string, string> SubstitutionPaths { get; set; }

            public PathUtility(string baseDir)
            {
                BaseDirectory = baseDir;
                SubstitutionPaths = new Dictionary<string, string>();
            }
            public PathUtility(string baseDir, IDictionary<string, string> substitutionPaths)
            {
                BaseDirectory = baseDir;
                SubstitutionPaths = substitutionPaths;
            }

            public string ConvertToRelativePath(string path)
            {
                if (Path.IsPathRooted(path))
                {
                    return MakeRelativePath(path);
                }
                else
                {
                    return path;
                }
            }

            public string ConvertToAbsolutePath(string path)
            {
                return Path.GetFullPath(Path.Combine(BaseDirectory, path));
            }

            public string ConvertToSubstitutedPath(string path)
            {
                string substitutedPath;
                ConvertToSubstitutedPath(path, out substitutedPath);
                return substitutedPath;
            }

            public bool ConvertToSubstitutedPath(string path, out string substitutedPath)
            {
                bool substituted = false;
                substitutedPath = path;

                // 置き換えるパスの長さが長いものを優先して使う
                var substitutionPaths = SubstitutionPaths.Values.OrderByDescending(x => x.Length);

                foreach (var substitutionPath in substitutionPaths)
                {
                    if (substitutedPath.IndexOf(substitutionPath) >= 0)
                    {
                        string key;
                        try
                        {
                            key = SubstitutionPaths.Single(x => x.Value == substitutionPath).Key;
                        }
                        catch (InvalidOperationException ex)
                        {
                            throw new InvalidOperationException(
                                string.Format(
                                    "パスの置換候補が複数存在するため、適切な置換を行うことができません。候補 = {0}",
                                    string.Join(", ", SubstitutionPaths.Where(x => x.Value == substitutionPath))),
                                ex);
                        }

                        substitutedPath = substitutedPath.Replace(substitutionPath, key);
                        substituted = true;
                    }
                }
                return substituted;
            }

            public string ConvertFromSubstitutedPath(string path)
            {
                string originalPath;
                ConvertFromSubstitutedPath(path, out originalPath);
                return originalPath;
            }

            public bool ConvertFromSubstitutedPath(string path, out string originalPath)
            {
                bool substituted = false;
                originalPath = path;
                foreach (var key in SubstitutionPaths.Keys)
                {
                    if (originalPath.IndexOf(key) >= 0)
                    {
                        var substitutionPath = SubstitutionPaths[key];
                        originalPath = originalPath.Replace(key, substitutionPath);
                        substituted = true;
                    }
                }
                return substituted;
            }

            public static bool IncludesMsbuildProperty(string path)
            {
                return s_MsbuildPropertyRegex.IsMatch(path);
            }

            #region Nact から実装をコピペ

            private string MakeRelativePath(string path)
            {
                var isFromAbs = Path.IsPathRooted(BaseDirectory);
                var isTargetAbs = Path.IsPathRooted(path);

                if (isFromAbs == isTargetAbs)
                {
                    var normalizedFrom = NormalizePath(BaseDirectory);
                    var normalizedTarget = NormalizePath(path);

                    if (isFromAbs)
                    {
                        // from と target が共に絶対パスの場合
                        //   → ドライブが異なるなら target をそのまま返す
                        //      そうでなければドライブを除いて相対パスとして扱う

                        var fromRoot = Path.GetPathRoot(normalizedFrom);
                        var targetRoot = Path.GetPathRoot(normalizedTarget);

                        if (fromRoot != targetRoot)
                        {
                            return normalizedTarget;
                        }

                        normalizedFrom = normalizedFrom.Substring(fromRoot.Length);
                        normalizedTarget = normalizedTarget.Substring(targetRoot.Length);
                    }

                    // from と target が共に相対パスの場合
                    {
                        var fromParts = normalizedFrom.Split('\\');
                        var targetParts = normalizedTarget.Split('\\');

                        var minNumParts = Math.Min(fromParts.Length, targetParts.Length);
                        int numMatch = 0;

                        for (numMatch = 0; numMatch < minNumParts; ++numMatch)
                        {
                            if (fromParts[numMatch] != targetParts[numMatch])
                            {
                                break;
                            }
                        }

                        var up = MakeRepeatString("..\\", fromParts.Length - numMatch);
                        var down = string.Join("\\", targetParts.Skip(numMatch));

                        var join = up + down;

                        return (join.Length == 0) ? "." : join;
                    }
                }

                // それ以外
                //   → エラー

                throw new ArgumentException("相対パスが作成できません。",
                    string.Format("基点={0}\n対象={1}\n", BaseDirectory, path));
            }

            private static Regex s_DoubleBackSlash = new Regex(@"\\+", RegexOptions.Compiled);
            private static Regex s_DriveLetter = new Regex(@"[a-z]:", RegexOptions.Compiled);
            private static Regex s_Dotdot = new Regex(@"([^\\]+)\\\.\.\\", RegexOptions.Compiled);

            private static string NormalizePath(string path)
            {
                // スラッシュをバックスラッシュに
                var p1 = path.Replace('/', '\\');

                // バックスラッシュの連続を一つのバックスラッシュに
                var p2 = s_DoubleBackSlash.Replace(p1, @"\");

                // ドライブ文字を大文字に
                var p3 = s_DriveLetter.Replace(p2, x => x.Value.ToUpper());

                // . を消す
                // TODO: UNC の場合にローカルホストを消してしまう
                var p4 = p3.Replace(@"\.\", @"\");

                // .. を畳む
                var p5 = p4;
                {
                    int start = 0;

                    for (;;)
                    {
                        bool isMatch = false;
                        p5 = s_Dotdot.Replace(p5, (m) =>
                        {
                            isMatch = true;
                            if (m.Groups[1].Value == "..")
                            {
                                start = m.Index + 2;
                                return m.Value;
                            }
                            else
                            {
                                return string.Empty;
                            }
                        }, 1, start);

                        if (!isMatch)
                        {
                            break;
                        }
                    }
                }
                if (Path.IsPathRooted(path) && !Path.IsPathRooted(p5))
                {
                    throw new ArgumentException(
                        "パスに含まれる .. が多すぎます。", string.Format("パス={0}\n", path));
                }

                return p5;
            }

            private static string MakeRepeatString(string one, int num)
            {
                var sb = new StringBuilder(one.Length * num);
                for (int i = 0; i < num; ++i)
                {
                    sb.Append(one);
                }
                return sb.ToString();
            }

            #endregion
        }
        #endregion
    }
}
