﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.ClassSystem;
using Nintendo.Nact.NactInterop;
using Nintendo.Nact.Values;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Diagnostics;
using static Nintendo.Nact.Extensions.NactValueExtensions;

namespace SigloNact.BuiltIns.BuildSystem
{
    [NactFunctionContainer]
    public static class ConfigurableValueFunctions
    {
        [NactFunction]
        public static IReadOnlyCollection<object> EvaluateConfigurableValueListImpl(
            NactFunctionContext functionContext,
            ValueFunction evaluateConfigurableInstance,
            object config,
            object value)
        {
            return ConfigurableValueEvaluator.Evaluate(
                functionContext,
                evaluateConfigurableInstance,
                config,
                value);
        }

        private class ConfigurableValueEvaluator
        {
            private readonly List<object> _buffer;
            private readonly NactFunctionContext _functionContext;
            private readonly ValueFunction _evaluateConfigurableInstance;
            private readonly object _config;

            public IReadOnlyCollection<object> Result => _buffer;

            public ConfigurableValueEvaluator(
                NactFunctionContext functionContext,
                ValueFunction evaluateConfigurableInstance,
                object config)
            {
                this._buffer = new List<object>();
                this._functionContext = functionContext;
                this._evaluateConfigurableInstance = evaluateConfigurableInstance;
                this._config = config;
            }

            /*
                EvaluateConfigurableValueListImpl は、Nact 言語では次の通り定義される関数です:

                function EvaluateConfigurableValueListImpl(evaluateConfigurableInstance, config, value) =
                    if !IsDefined(value) then
                        undefined
                    elif IsList(value) then
                        // NOTE: この SelectMany により結果は flat になる
                        mergeEvaluatedValuesIfListOfMaps(
                            value.SelectMany(
                                x => EvaluateConfigurableValueListImpl(evaluateConfigurableInstance, config, x)))
                    elif IsMap(value) then
                        value.SelectValue(
                            x => EvaluateConfigurableValueListImpl(evaluateConfigurableInstance, config, x))]
                    elif IsClosure(value) then
                        ToList(value(config))
                    elif IsInstance(value) then
                        [evaluateConfigurableInstance(config, value)]
                    else
                        [value]
                    end;

                function mergeEvaluatedValuesIfListOfMaps(list) =
                    if list.Length != 0 && list.All(x => IsMap(x)) then
                        if list.Length == 1 then
                            list
                        else
                            [Aggregate(
                                list,
                                Map(),
                                // リストの末尾側のマップ値を優先して使用する
                                (total, x) => MergeMap(total, x, (k, oldValue, newValue) => newValue))]
                        end
                    else
                        list
                    end;
            */
            public static IReadOnlyCollection<object> Evaluate(
                NactFunctionContext functionContext,
                ValueFunction evaluateConfigurableInstance,
                object config,
                object value)
            {
                if (value == null)
                {
                    return null;
                }

                var evaluator = new ConfigurableValueEvaluator(
                    functionContext,
                    evaluateConfigurableInstance,
                    config);
                evaluator.EvaluateAndAdd(value);
                return evaluator.Result;
            }

            public void EvaluateAndAdd(object value)
            {
                switch (value)
                {
                    case NactList list:
                        {
                            if (list.Count == 0)
                            {
                                return;
                            }

                            // 各要素を評価したリストを flatten したものを追加する
                            int firstIndex = _buffer.Count;
                            foreach (var innerValue in list)
                            {
                                EvaluateAndAdd(innerValue);
                            }

                            // 追加した要素が全て NactMap であればそれらをマージする
                            if (_buffer.Count - firstIndex > 1)
                            {
                                MergeIfListOfMaps(firstIndex);
                            }
                        }
                        return;
                    case NactMap map:
                        {
                            _buffer.Add(map.SelectValues(EvaluateToSeparateList));
                        }
                        return;
                    case ValueFunction func:
                        {
                            AddExpandingList(_functionContext.Evaluate(func, ImmutableArray.Create(_config)));
                        }
                        return;
                    case IInstance ins:
                        {
                            var list = _functionContext
                                .Evaluate(_evaluateConfigurableInstance, ImmutableArray.Create(_config, ins))
                                .CheckedCast<NactList>();
                            _buffer.AddRange(list);
                        }
                        return;
                    default:
                        {
                            Debug.Assert(!(value is NactList));
                            _buffer.Add(value);
                        }
                        return;
                }
            }

            private IReadOnlyCollection<object> EvaluateToSeparateList(object value)
            {
                return Evaluate(
                    _functionContext,
                    _evaluateConfigurableInstance,
                    _config,
                    value);
            }

            private void AddExpandingList(object value)
            {
                if (value is NactList list)
                {
                    _buffer.AddRange(list);
                }
                else
                {
                    _buffer.Add(value);
                }
            }

            // firstIndex 以降の全ての要素が NactMap であればそれらをマージする
            private void MergeIfListOfMaps(int firstIndex)
            {
                for (int i = firstIndex; i < _buffer.Count; i++)
                {
                    if (!(_buffer[i] is NactMap))
                    {
                        return;
                    }
                }

                // リストの末尾側のマップ値を優先して使用する
                var mergedMap = new Dictionary<string, object>();
                for (int i = _buffer.Count - 1; i >= firstIndex; i --)
                {
                    foreach (var kv in (NactMap)_buffer[i])
                    {
                        if (!mergedMap.ContainsKey(kv.Key))
                        {
                            mergedMap.Add(kv.Key, kv.Value);
                        }
                    }
                }

                _buffer.RemoveRange(firstIndex, _buffer.Count - firstIndex);
                _buffer.Add(NactValueConverter.ToNactValue(typeof(IReadOnlyDictionary<string, object>), mergedMap));
            }
        }
    }
}
