﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

#r "System.Xml"
#r "System.Xml.Linq"
#r "System.IO.Compression"
#r "System.IO.Compression.FileSystem"

using System;
using System.IO;
using System.IO.Compression;
using System.Text.RegularExpressions;
using System.Xml.Linq;

string[] GetEntries(string archivePath)
{
    using (var zip = ZipFile.Open(archivePath, ZipArchiveMode.Read))
    {
        return zip.Entries.Select(x => x.FullName).ToArray();
    }
}

(string path, string id, string version) GetNupkgIdAndVersion(string path)
{
    var name = Path.GetFileNameWithoutExtension(path);
    var m = Regex.Match(name, @"^(([a-zA-Z_][a-zA-Z_0-9]*\.)+)\d");
    if (!m.Success)
    {
        throw new Exception($"Invalid nupkg name : {name}");
    }

    var id = m.Groups[1].Value.TrimEnd('.');
    var version = name.Substring(id.Length + 1);

    return (path, id, version);
}

void RewritePackageReferences(string csprojPath, (string path, string id, string version)[] nupkgs)
{
    Console.WriteLine($"Updating {csprojPath}...");

    var xml = XDocument.Load(csprojPath);
    var ns = xml.Root.Name.Namespace;
    foreach (var packageReferenceElement in xml.Descendants(ns + "PackageReference"))
    {
        var id = packageReferenceElement.Attribute("Include").Value;
        var nupkg = nupkgs.SingleOrDefault(x => string.CompareOrdinal(id, x.id) == 0);
        if (nupkg.id != default(string))
        {
            var versionElement = packageReferenceElement.Element(ns + "Version");
            versionElement.SetValue(nupkg.version);
            Console.WriteLine($"    {id} -> {nupkg.version}");
        }
    }
    xml.Save(csprojPath);
}

void GenerateNactVersionText(string path, string sigloNactVersion)
{
    Console.WriteLine($"Updating {path}...");

    // CRLF を書くようにする
    File.WriteAllLines(path, new string[] { sigloNactVersion });
}

void GenerateFilesNarl(string path, string memberName, string[] files)
{
    Console.WriteLine($"Updating {path}...");

    var lines = new List<string>();

    lines.Add($"@private {memberName} = [");
    foreach (var file in files)
    {
        lines.Add($"    \"{file}\",");
    }
    lines.Add("];");

    File.WriteAllLines(path, lines, new UTF8Encoding(true));
}

void GenerateNactBundle(string path, string[] pluginFiles, string[] sigloNactFiles)
{
    Console.WriteLine($"Updating {path}...");

    var lines = new List<string>()
    {
        "Bundle:",
        "  Type: Private",
        "  Files:",
        "    #---------------------------------------------------------------------------",
        "    # Nact",
        "    #---------------------------------------------------------------------------",
        "    - Integrate/CommandLineTools/nact.exe",
        "    - Integrate/CommandLineTools/NactVersion.txt",
        "",
    };

    foreach (var file in pluginFiles)
    {
        var relativePath = "Build/Nact/NactBackEndPlugins/" + file.Replace('\\', '/');
        lines.Add($"    - {relativePath}");
    }

    lines.Add("");

    foreach (var file in sigloNactFiles)
    {
        var relativePath = "Build/Nact/" + file.Replace('\\', '/');
        lines.Add($"    - {relativePath}");
    }

    File.WriteAllLines(path, lines, new UTF8Encoding(true));
}

void RemoveExistingNugetPackages(string sigloRoot, string id)
{
    var sigloNuGetFeed = Path.Combine(sigloRoot, @"Externals\NuGetFeed");

    Console.WriteLine($"Removing existing {id} packages...");

    var packageDir = Path.Combine(sigloNuGetFeed, id);

    if (!Directory.Exists(packageDir))
    {
        return;
    }

    foreach (var subDir in Directory.GetDirectories(packageDir))
    {
        var subDirName = Path.GetFileName(subDir);
        var filesToRemove = new string[]
        {
            Path.Combine(subDir, $"{id}.{subDirName}.nupkg"),
            Path.Combine(subDir, $"{id}.{subDirName}.nupkg.sha512"),
            Path.Combine(subDir, $"{id}.nuspec"),
        };

        // 安全のため、想定するファイルが全部存在する場合に限り、それらファイルだけを削除する
        if (filesToRemove.All(x => File.Exists(x)))
        {
            foreach (var file in filesToRemove)
            {
                Console.WriteLine($"    Removing {file}...");
                File.Delete(file);
            }
        }
    }
}

void AddNugetPackage(string sigloRoot, string path)
{
    var scriptPath = Path.Combine(sigloRoot, @"Integrate\Scripts\Add-SigloNuGetPackages.ps1");

    ExecuteProgram(
        "powershell.exe",
        $"-Version 2.0  -ExecutionPolicy RemoteSigned -file \"{scriptPath}\" -Force \"{path}\"");
}

void ExecuteProgram(string fileName, string arguments)
{
    System.Console.WriteLine();
    System.Console.WriteLine($"Execute {fileName} {arguments}");

    var psi = new ProcessStartInfo()
    {
        FileName = fileName,
        Arguments = arguments,
        WorkingDirectory = null,
        UseShellExecute = false,
        CreateNoWindow = false,
        RedirectStandardError = false,
        RedirectStandardInput = false,
        RedirectStandardOutput = false,
    };

    using (var p = Process.Start(psi))
    {
        p.WaitForExit();
        if (p.ExitCode != 0)
        {
            throw new Exception("Failed");
        }
    }

    System.Console.WriteLine();
}

//
// main
//
if (Args.Count < 2)
{
    Console.Error.WriteLine("DeployNact <siglo root> <extracted artifact directory>");
    return 1;
}

var sigloRoot = Args[0];
var artifactArchiveDirectory = Args[1];

//
// artifactArchiveDirectory 中のファイルを収集
//
var zipFiles = new List<string>();
var nupkgFiles = new List<string>();
foreach (var archivePath in Directory.GetFiles(artifactArchiveDirectory))
{
    switch (Path.GetExtension(archivePath))
    {
        case ".zip":
            zipFiles.Add(archivePath);
            break;
        case ".nupkg":
            nupkgFiles.Add(archivePath);
            break;
    }
}

//
// もろもろのデータを収集
//
var pluginFiles = new string[] {
    @"AdlCommonLibrary.dll",
    @"ja\SigloNact.Resources.resources.dll",
    @"ResultLibrary.dll",
    @"SigloNact.dll",
    @"SigloNact.Resources.dll",
};

var configFilesForSiglo = new string[]
{
    "nact.Siglo.config",
    "nactSettings.Siglo.xml",
};

// プラグイン以外が使う想定の NuGet パッケージは手動で更新する
var ignoredNupkgIds = new string[]
{
    "Nintendo.Nact.Utilities.Base",
    "Nintendo.Nact.Utilities.MSBuild",
    "Nintendo.Nact.Utilities.VisualStudio",
    "Nintendo.Nact.Utilities.Win32",
};

var zipFile = zipFiles.Single();
var sigloNactVersion = Path.GetFileNameWithoutExtension(zipFile);
var sigloNactFiles = GetEntries(zipFile);
var nupkgs = nupkgFiles
    .Select(x => GetNupkgIdAndVersion(x))
    .Where(x => !ignoredNupkgIds.Contains(x.id))
    .ToArray();

Console.WriteLine($"Source directory :");
Console.WriteLine($"    {artifactArchiveDirectory}");
Console.WriteLine($"SigloNact :");
Console.WriteLine($"    {sigloNactVersion}");
Console.WriteLine($"NuGet packages :");
foreach (var nupkg in nupkgs)
{
    Console.WriteLine($"    {nupkg.id} {nupkg.version}");
}
Console.WriteLine();

//
// ファイルを生成
//
foreach (var nupkg in nupkgs)
{
    RemoveExistingNugetPackages(sigloRoot, nupkg.id);
}

foreach (var nupkg in nupkgs)
{
    AddNugetPackage(sigloRoot, nupkg.path);
}

RewritePackageReferences(
    Path.Combine(sigloRoot, @"Programs\Alice\Sources\Tools\SigloNact\SigloNact.Test\SigloNact.Test.csproj"),
    nupkgs);

RewritePackageReferences(
    Path.Combine(sigloRoot, @"Programs\Alice\Sources\Tools\SigloNact\SigloNact\SigloNact.csproj"),
    nupkgs);

GenerateNactVersionText(
    Path.Combine(sigloRoot, @"Integrate\CommandLineTools\NactVersion.txt"),
    sigloNactVersion);

GenerateFilesNarl(
    Path.Combine(sigloRoot, @"Programs\Alice\Sources\Tools\SigloNact\PluginFiles.autogen.narl"),
    "sigloNactPluginFiles",
    pluginFiles);

GenerateFilesNarl(
    Path.Combine(sigloRoot, @"Programs\Alice\Sources\Tools\SigloNact\SigloNactFiles.autogen.narl"),
    "sigloNactFiles",
    sigloNactFiles);

GenerateNactBundle(
    Path.Combine(sigloRoot, @"Integrate\Packages\BundleDefinitions\Nact.bundle.yml"),
    pluginFiles,
    sigloNactFiles.Where(x => !configFilesForSiglo.Contains(Path.GetFileName(x))).ToArray());

// dev, feature ブランチ産でも LocalBuild でもなければ共有ディレクトリにコピー
if (!zipFile.Contains("commit") && !zipFile.Contains("LocalBuild"))
{
    var destination = Path.Combine(@"\\devdatasrv2\share-siglo$\ForAll\setup\SigloNact", Path.GetFileName(zipFile));
    if (!File.Exists(destination))
    {
        Console.WriteLine($"Copying to {destination}...");
        File.Copy(zipFile, destination);
        new FileInfo(destination).Attributes |= FileAttributes.ReadOnly;
    }
}
