﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using YamlDotNet.Serialization;

namespace Nintendo.MakeVisualStudioProject
{
    public enum VcProjectConfigurationType
    {
        StaticLibrary,
        DynamicLibrary,
        Application
    }

    public class ProjectSetting
    {
        public string Name { get; set; }
        public IEnumerable<BuildTargetSetting> TargetSettings { get; set; }
        public VcProjectConfigurationType ConfigurationType { get; set; }
        public string Keyword { get; set; }
        public string ProjectGuid { get; set; }
        public string RootNamespace { get; set; }

        public string LocalPropertySheetName { get; set; }
        public string ProjectConfigurationPropertySheetName { get; set; }
        public string DefaultDefinitionsPropertySheetName { get; set; }
        public string CommonDefinitionsPropertySheetName { get; set; }

        public static ProjectSetting Read(string path)
        {
            ProjectSetting setting;
            using (var stream = new StreamReader(path))
            {
                var deserializer = new Deserializer();
                setting = deserializer.Deserialize<ProjectSetting>(stream);
            }

            return setting;
        }
        public static void Write(ProjectSetting setting, string path)
        {
            using (var stream = new StreamWriter(path))
            {
                var serializer = new Serializer();
                serializer.Serialize(stream, setting);
            }
        }

        public ProjectSetting()
        {
            SetDefaultValues();
        }

        private void SetDefaultValues()
        {
            Keyword = "Win32Proj";
        }
    }

    internal static class ProjectSettingUtility
    {
        public static IEnumerable<T> EnsureNonNullEnumerable<T>(IEnumerable<T> enumerable)
        {
            return enumerable ?? Enumerable.Empty<T>();
        }

        public static IEnumerable<string> EnsureNonNullEnumerable(IEnumerable<string> enumerable)
        {
            return enumerable?.Select(x => x == null ? string.Empty : x) ?? Enumerable.Empty<string>();
        }
    }

    public class BuildTargetSetting
    {
        public string Platform { get; set; }
        public string Configuration { get; set; }
        public string PrecompiledHeaderFile { get; set; }
        public string PreBuildEvent { get; set; }
        public string PreLinkEvent { get; set; }
        public string PostBuildEvent { get; set; }
        public string LinkerType { get; set; }
        public string DescFile { get; set; }
        public string MetaFile { get; set; }
        public string ApplicationDataDirectory { get; set; }
        public string NroDeployDirectory { get; set; }
        public string NrrFileName { get; set; }

        // IEnumerable の null チェックが面倒なので、絶対に null を返さない仕様にしておく
        private IEnumerable<SourceFile> m_SourceFiles = Enumerable.Empty<SourceFile>();
        private IEnumerable<string> m_HeaderFiles = Enumerable.Empty<string>();
        private IEnumerable<string> m_UserFiles = Enumerable.Empty<string>();
        private IEnumerable<string> m_CompileOptions = Enumerable.Empty<string>();
        private IEnumerable<string> m_PreprocessorMacros = Enumerable.Empty<string>();
        private IEnumerable<string> m_IncludeSearchPath = Enumerable.Empty<string>();
        private IEnumerable<string> m_ObjectFiles = Enumerable.Empty<string>();
        private IEnumerable<string> m_ArchiveOptions = Enumerable.Empty<string>();
        private IEnumerable<string> m_LibraryFiles = Enumerable.Empty<string>();
        private IEnumerable<string> m_LibraryNames = Enumerable.Empty<string>();
        private IEnumerable<string> m_LibrarySearchPath = Enumerable.Empty<string>();
        private IEnumerable<string> m_LinkOptions = Enumerable.Empty<string>();
        private IEnumerable<string> m_SdkDynamicLinkLibraryNames = Enumerable.Empty<string>();
        private IEnumerable<string> m_ManualLoadDynamicLinkLibraryNames = Enumerable.Empty<string>();
        private IEnumerable<string> m_PropertySheets = Enumerable.Empty<string>();
        private IEnumerable<ProjectProperty> m_ConfigurationProperties = Enumerable.Empty<ProjectProperty>();
        private IEnumerable<ProjectProperty> m_PathProperties = Enumerable.Empty<ProjectProperty>();

        public IEnumerable<SourceFile> SourceFiles
        {
            get
            {
                return m_SourceFiles;
            }
            set
            {
                m_SourceFiles = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> HeaderFiles
        {
            get
            {
                return m_HeaderFiles;
            }
            set
            {
                m_HeaderFiles = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> UserFiles
        {
            get
            {
                return m_UserFiles;
            }
            set
            {
                m_UserFiles = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> CompileOptions
        {
            get
            {
                return m_CompileOptions;
            }
            set
            {
                m_CompileOptions = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> PreprocessorMacros
        {
            get
            {
                return m_PreprocessorMacros;
            }
            set
            {
                m_PreprocessorMacros = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> IncludeSearchPath
        {
            get
            {
                return m_IncludeSearchPath;
            }
            set
            {
                m_IncludeSearchPath = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> ObjectFiles
        {
            get
            {
                return m_ObjectFiles;
            }
            set
            {
                m_ObjectFiles = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> ArchiveOptions
        {
            get
            {
                return m_ArchiveOptions;
            }
            set
            {
                m_ArchiveOptions = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> LibraryFiles
        {
            get
            {
                return m_LibraryFiles;
            }
            set
            {
                m_LibraryFiles = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> LibraryNames
        {
            get
            {
                return m_LibraryNames;
            }
            set
            {
                m_LibraryNames = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> LibrarySearchPath
        {
            get
            {
                return m_LibrarySearchPath;
            }
            set
            {
                m_LibrarySearchPath = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> LinkOptions
        {
            get
            {
                return m_LinkOptions;
            }
            set
            {
                m_LinkOptions = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> SdkDynamicLinkLibraryNames
        {
            get
            {
                return m_SdkDynamicLinkLibraryNames;
            }
            set
            {
                m_SdkDynamicLinkLibraryNames = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> ManualLoadDynamicLinkLibraryNames
        {
            get
            {
                return m_ManualLoadDynamicLinkLibraryNames;
            }
            set
            {
                m_ManualLoadDynamicLinkLibraryNames = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<string> PropertySheets
        {
            get
            {
                return m_PropertySheets;
            }
            set
            {
                m_PropertySheets = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<ProjectProperty> ConfigurationProperties
        {
            get
            {
                return m_ConfigurationProperties;
            }
            set
            {
                m_ConfigurationProperties = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
        public IEnumerable<ProjectProperty> PathProperties
        {
            get
            {
                return m_PathProperties;
            }
            set
            {
                m_PathProperties = ProjectSettingUtility.EnsureNonNullEnumerable(value);
            }
        }
    }

    public class SourceFile : IEquatable<SourceFile>
    {
        public string Path { get; set; }

        // IEnumerable の null チェックが面倒なので、絶対に null を返さない仕様にしておく
        private IEnumerable<string> m_CompileOptions = Enumerable.Empty<string>();

        public IEnumerable<string> CompileOptions
        {
            get
            {
                return m_CompileOptions;
            }
            set
            {
                m_CompileOptions = ProjectSettingUtility.EnsureNonNullEnumerable(value).OrderBy(op => op);
            }
        }

        #region IEquatable 関係
        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }
            else if (obj is SourceFile)
            {
                return Equals((SourceFile)obj);
            }
            else
            {
                return false;
            }
        }
        public bool Equals(SourceFile other)
        {
            if (other == null)
            {
                return false;
            }
            else
            {
                return Path == other.Path && (CompileOptions != null && other.CompileOptions != null) && CompileOptions.SequenceEqual(other.CompileOptions);
            }
        }
        public static bool operator == (SourceFile lhs, SourceFile rhs)
        {
            if ((object)lhs == null || (object)rhs == null)
            {
                return object.ReferenceEquals(lhs, rhs);
            }
            else
            {
                return lhs.Equals(rhs);
            }
        }
        public static bool operator != (SourceFile lhs, SourceFile rhs)
        {
            if ((object)lhs == null || (object)rhs == null)
            {
                return !object.ReferenceEquals(lhs, rhs);
            }
            else
            {
                return !lhs.Equals(rhs);
            }
        }
        public override int GetHashCode()
        {
            var pathHash = Path != null ? Path.GetHashCode() : 0;
            var optionsHash = CompileOptions != null ? CompileOptions.Select(op => op.GetHashCode()).Aggregate(0, (lhs, rhs) => lhs ^ rhs) : 0;
            return pathHash ^ optionsHash;
        }
        #endregion
    }

    public class ProjectProperty : IEquatable<ProjectProperty>
    {
        public string Name { get; set; }
        public string Value { get; set; }

        #region IEquatable 関係
        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }
            else if (obj is ProjectProperty)
            {
                return Equals((ProjectProperty)obj);
            }
            else
            {
                return false;
            }
        }
        public bool Equals(ProjectProperty other)
        {
            if (other == null)
            {
                return false;
            }
            else
            {
                return Name == other.Name && Value == other.Value;
            }
        }
        public static bool operator == (ProjectProperty lhs, ProjectProperty rhs)
        {
            if ((object)lhs == null || (object)rhs == null)
            {
                return object.ReferenceEquals(lhs, rhs);
            }
            else
            {
                return lhs.Equals(rhs);
            }
        }
        public static bool operator != (ProjectProperty lhs, ProjectProperty rhs)
        {
            if ((object)lhs == null || (object)rhs == null)
            {
                return !object.ReferenceEquals(lhs, rhs);
            }
            else
            {
                return !lhs.Equals(rhs);
            }
        }
        public override int GetHashCode()
        {
            var nameHash = Name != null ? Name.GetHashCode() : 0;
            var valueHash = Value != null ? Value.GetHashCode() : 0;
            return nameHash ^ valueHash;
        }
        #endregion
    }
}
