﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.MakeVisualStudioProject
{
    // 実装は一林さん謹製
    // ssh://git@spdlybra.nintendo.co.jp:7999/~ichibayashi_hironori/ichilib.git より取得可能 (2015.07.30)

    // UUID 仕様:
    // http://www.ietf.org/rfc/rfc4122.txt
    // http://www.rfc-editor.org/errata_search.php?rfc=4122&eid=1352

    /// <summary>
    /// Guid の操作を行うユーティリティ
    /// </summary>
    public static class GuidUtility
    {
        /// <summary>
        /// RFC4122 4.3 の方法で、UUID Version 3 を生成します。
        /// </summary>
        /// <param name="name">UUID 生成に使用する name</param>
        /// <param name="uuidNameSpace">UUID name space</param>
        /// <returns>生成された Guid クラスのインスタンス</returns>
        public static Guid CreateGuidFromNameMd5(Guid uuidNameSpace, byte[] name)
        {
            using (var generator = new MD5Cng())
            {
                return CreateGuidFromNameWithHashAlgorithm(uuidNameSpace, name, generator, 3);
            }
        }

        /// <summary>
        /// RFC4122 4.3 の方法で、UUID Version 5 を生成します。
        /// </summary>
        /// <param name="name">UUID 生成に使用する name</param>
        /// <param name="uuidNameSpace">UUID name space</param>
        /// <returns>生成された Guid クラスのインスタンス</returns>
        public static Guid CreateGuidFromNameSha1(Guid uuidNameSpace, byte[] name)
        {
            using (var generator = new SHA1Cng())
            {
                return CreateGuidFromNameWithHashAlgorithm(uuidNameSpace, name, generator, 5);
            }
        }

        // 指定したハッシュアルゴリズム、バージョンを用いて、RFC4122 4.3 の方法で UUID を生成する
        private static Guid CreateGuidFromNameWithHashAlgorithm(Guid uuidNameSpace, byte[] name, HashAlgorithm generator, byte version)
        {
            var nameSpaceUuidByteArray = ToUuidByteArray(uuidNameSpace);

            var byteArrayToHash = new List<byte>(nameSpaceUuidByteArray.Length + name.Length);
            byteArrayToHash.AddRange(nameSpaceUuidByteArray);
            byteArrayToHash.AddRange(name);

            var hash = generator.ComputeHash(byteArrayToHash.ToArray());
            var uuidByteArray = new byte[16];
            Array.Copy(hash, uuidByteArray, 16);
            // Version
            uuidByteArray[6] = (byte)((hash[6] & 0x0f) | ((version << 4) & 0xf0));
            // Variant = 10b
            uuidByteArray[8] = (byte)((hash[8] & 0x3f) | 0x80);

            return FromUuidByteArray(uuidByteArray);
        }

        /// <summary>
        /// Guid を UUID のバイト表現に変換します。
        /// すなわち、GUID の各フィールドをネットワークバイトオーダーで格納したバイト配列を返します。
        /// 参照) RFC4122 Section 4
        /// </summary>
        /// <param name="guid">変換元の Guid クラスのインスタンス</param>
        /// <returns>UUID のバイト表現</returns>
        public static byte[] ToUuidByteArray(Guid guid)
        {
            byte[] byteArray = guid.ToByteArray();
            SwapBetweenGuidAndUuid(byteArray);
            return byteArray;
        }

        /// <summary>
        /// UUID のバイト表現を Guid に変換します。
        /// </summary>
        /// <param name="uuidByteArray">UUID のバイト表現</param>
        /// <returns>Guid クラスのインスタンス</returns>
        public static Guid FromUuidByteArray(byte[] uuidByteArray)
        {
            byte[] byteArray = new byte[uuidByteArray.Length];
            Array.Copy(uuidByteArray, byteArray, byteArray.Length);
            SwapBetweenGuidAndUuid(byteArray);
            return new Guid(byteArray);
        }

        // GUID と UUID 間のバイトオーダー反転を行う
        private static void SwapBetweenGuidAndUuid(byte[] byteArray)
        {
            // GUID と UUID では最初の 3 つのフィールドのバイトオーダーが逆であるため、反転します。
            Array.Reverse(byteArray, 0, 4);
            Array.Reverse(byteArray, 4, 2);
            Array.Reverse(byteArray, 6, 2);
        }
    }
}
