﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Microsoft.Build.Construction;

namespace Nintendo.MakeVisualStudioProject.Generator
{
    internal class ProjectFiltersGenerator
    {
        private static readonly Dictionary<string, IEnumerable<string>> s_FilterNameAndExtensions = new Dictionary<string, IEnumerable<string>>()
        {
            { "Source Files", new string[]
                { "cpp", "c", "cc", "cxx", "def", "odl", "idl", "hpj", "bat", "asm", "asmx" }
            },
            { "Header Files", new string[]
                { "h", "hpp", "hxx", "hm", "inl", "inc", "xsd" }
            },
            { "Resource Files", new string[]
                { "rc", "ico", "cur", "bmp", "dlg", "rc2", "rct", "bin", "rgs", "gif", "jpg", "jpeg", "jpe", "resx", "tiff", "tif", "png", "wav", "mfcribbon-ms" }
            },
            { "Shader Files", new string[]
                { "gsh", "csh", "vsh", "psh", "fsh", "glsl", "hlsl" }
            },
            { "Build", new string[]
                { "nact", "props", "targets" }
            },
        };
        private static readonly Guid s_ProjectFilterGuidNamespace = new Guid("7548ea01-6852-475b-8aec-f58d9b894602");

        public static ProjectRootElement Generate(ProjectSetting setting, PathUtility pathUtility)
        {
            var projectRootElement = ProjectRootElement.Create();

            var clCompiles = setting.TargetSettings.SelectMany(x => x.SourceFiles).Select(x => pathUtility.ConvertToRelativePath(x.Path)).Distinct();
            var clIncludes = setting.TargetSettings.SelectMany(x => x.HeaderFiles).Select(x => pathUtility.ConvertToRelativePath(x)).Distinct();
            var nones = setting.TargetSettings.SelectMany(x => x.UserFiles).Select(x => pathUtility.ConvertToRelativePath(x)).Distinct();

            AddFilterDefinitions(projectRootElement, clCompiles, clIncludes, nones);
            AddItems(projectRootElement, "ClCompile", clCompiles);
            AddItems(projectRootElement, "ClInclude", clIncludes);
            AddItems(projectRootElement, "None", nones);

            return projectRootElement;
        }

        private static void AddFilterDefinitions(
            ProjectRootElement projectRootElement, IEnumerable<string> clCompiles, IEnumerable<string> clIncludes, IEnumerable<string> nones)
        {
            var filterNameList = new List<string>();

            foreach (var rootFilterName in s_FilterNameAndExtensions)
            {
                if (clCompiles.Any(x => rootFilterName.Value.Contains(Path.GetExtension(x).TrimStart('.'))) ||
                    clIncludes.Any(x => rootFilterName.Value.Contains(Path.GetExtension(x).TrimStart('.'))) ||
                    nones.Any(x => rootFilterName.Value.Contains(Path.GetExtension(x).TrimStart('.'))))
                {
                    filterNameList.Add(rootFilterName.Key);
                }
            }

            foreach (var filterName in ConvertToShortPath(clCompiles).Select(x => GenerateFilterName(x)).Where(x => !string.IsNullOrEmpty(x)))
            {
                filterNameList.AddRange(EnumerateNecessaryFilterNames(filterName));
            }
            foreach (var filterName in ConvertToShortPath(clIncludes).Select(x => GenerateFilterName(x)).Where(x => !string.IsNullOrEmpty(x)))
            {
                filterNameList.AddRange(EnumerateNecessaryFilterNames(filterName));
            }
            foreach (var filterName in ConvertToShortPath(nones).Select(x => GenerateFilterName(x)).Where(x => !string.IsNullOrEmpty(x)))
            {
                filterNameList.AddRange(EnumerateNecessaryFilterNames(filterName));
            }

            var filterNames = filterNameList.Distinct();

            var itemGroup = AddItemGroup(projectRootElement);

            foreach (var filterName in filterNames)
            {
                var filter = itemGroup.AddItem("Filter", filterName);
                filter.AddMetadata("UniqueIdentifier", GenerateUniqueIdentifier(filterName));
                if (s_FilterNameAndExtensions.ContainsKey(filterName))
                {
                    filter.AddMetadata("Extensions", string.Join(";", s_FilterNameAndExtensions[filterName]));
                }
            }
        }

        private static void MakeShortPathListImpl(List<string> result, string basePath, IEnumerable<IReadOnlyList<string>> paths)
        {
            // - A\B\C
            // - A\B\D\E
            // - X\Y
            // というディレクトリ構成から
            // - A%255cB\C
            // - A%255cB\D%255cE
            // - X%255cY
            // というリストを作る

            var group = paths.GroupBy(x => x.First());
            foreach (var data in group)
            {
                var commonCount = 0;

                // 共通部分の抜き出し
                while (data.All(e => e.Count > commonCount))
                {
                    var skiped = data.Select(e => e.Skip(commonCount).First());

                    if (skiped.Distinct().Count() != 1)
                    {
                        break;
                    }
                    commonCount++;
                }
                var commonDirs = data.First().Take(commonCount).ToArray();
                var commonDirName = string.Join("%255c", commonDirs);    // %255c は \ のエスケープ ( \ -> %5c -> %255c )


                var rest = data.Select(e => e.Skip(commonCount).ToArray()).Where(e => e.Length > 0).ToArray();
                if (rest.Length > 0)
                {
                    if (data.Any(x => x.SequenceEqual(commonDirs)))
                    {
                        result.Add(Path.Combine(basePath, commonDirName));
                    }

                    MakeShortPathListImpl(result, Path.Combine(basePath, commonDirName), rest);
                }
                else
                {
                    result.Add(Path.Combine(basePath, commonDirName));
                }
            }
        }

        private static List<string> MakeShortPathList(IEnumerable<IReadOnlyList<string>> directoryPaths)
        {
            var list = new List<string>();
            MakeShortPathListImpl(list, "", directoryPaths);
            return list;
        }

        private static Dictionary<string, string> MakeShortPathMap(IEnumerable<string> paths)
        {
            var directoryPaths = paths.Select(path => Path.GetDirectoryName(path)).Distinct();
            var pathList = MakeShortPathList(directoryPaths.Select(path => path.Split('\\'))).ToArray();

            var map = new Dictionary<string, string>();

            foreach (var path in pathList)
            {
                map[path.Replace("%255c", "\\")] = path;
            }

            return map;
        }

        // 先頭から '.' のみの要素をスキップ
        private static string TrimRelativePathElement(string originalPath)
            => string.Join("\\", originalPath.Split('\\').SkipWhile(x => Regex.IsMatch(x, @"^\.+$")));

        // テストのために internal にする
        internal static IEnumerable<string> ConvertToShortPath(IEnumerable<string> paths)
        {
            var trimmedPath = paths.Select(x => TrimRelativePathElement(x)).ToArray();
            var map = MakeShortPathMap(trimmedPath);

            return trimmedPath.Select(path => Path.Combine(map[Path.GetDirectoryName(path)], Path.GetFileName(path)));
        }

        private static void AddItems(ProjectRootElement projectRootElement, string itemType, IEnumerable<string> paths)
        {
            var itemGroup = AddItemGroup(projectRootElement);

            var map = MakeShortPathMap(paths.Select(x => TrimRelativePathElement(x)));

            foreach (var path in paths)
            {
                var trimmedPath = TrimRelativePathElement(path);

                var trimmedFilename = Path.GetFileName(trimmedPath);
                var trimmedDirName = Path.GetDirectoryName(trimmedPath);

                var item = itemGroup.AddItem(itemType, path);

                var filterName = GenerateFilterName(Path.Combine(map[trimmedDirName], trimmedFilename));
                if (!string.IsNullOrEmpty(filterName))
                {
                    item.AddMetadata("Filter", filterName);
                }
            }
        }

        private static string GenerateFilterName(string path)
        {
            var extension = Path.GetExtension(path).TrimStart('.');

            var matchingFilter = s_FilterNameAndExtensions.Where(x => x.Value.Contains(extension));
            if (matchingFilter.Any())
            {
                var filterRootName = matchingFilter.Single().Key;
                var subFilterName = Path.GetDirectoryName(TrimRelativePathElement(path));

                if (string.IsNullOrEmpty(subFilterName))
                {
                    return filterRootName;
                }
                else
                {
                    return string.Format(@"{0}\{1}", filterRootName, subFilterName);
                }
            }
            else
            {
                return null;
            }
        }

        private static IEnumerable<string> EnumerateNecessaryFilterNames(string filterNames)
        {
            // aaa\bbb\ccc というフィルタを使用したい場合は、aaa\bbb\ccc だけではなく aaa と aaa\bbb の
            // フィルタ定義も必要となる (でなければ、必要なファイルが Visual Studio 上に表示されなくなる)
            //
            // 使用したいフィルタから実際に必要なフィルタをすべて列挙する

            var components = filterNames.Split('\\');

            for (var i = 1; i <= components.Length; i++)
            {
                yield return string.Join("\\", components.Take(i));
            }
        }

        private static ProjectItemGroupElement AddItemGroup(ProjectRootElement projectRootElement)
        {
            var ret = projectRootElement.CreateItemGroupElement();
            projectRootElement.AppendChild(ret);
            return ret;
        }

        private static string GenerateUniqueIdentifier(string filterName)
        {
            // フィルタ定義には GUID が必要なため、適当にフィルタ名から作る
            return string.Format("{{{0}}}", GuidUtility.CreateGuidFromNameMd5(
                s_ProjectFilterGuidNamespace, Encoding.UTF8.GetBytes(filterName)));
        }
    }
}
