﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Microsoft.Build.Construction;
using Nintendo.Foundation.IO;
using Nintendo.MakeVisualStudioProject.Converter;
using YamlDotNet.Serialization;

namespace Nintendo.MakeVisualStudioProject.Generator
{
    internal class GenerateCommand : ICommand
    {
        public class CommandParameter : ICommandParameter
        {
            private string m_OutputFilePath;
            private bool m_CreateFilters;

            [CommandLineValue(
                0,
                Description = "プロジェクト設定ファイルのパスです。",
                IsRequired = true,
                ValueName = "yml")]
            public string InputFilePath { get; private set; }

            [CommandLineOption(
                'o',
                Description = "出力するプロジェクトファイルのパスです。",
                ValueName = "vcxproj")]
            public string OutputFilePath
            {
                get
                {
                    if (m_OutputFilePath != null)
                    {
                        return m_OutputFilePath;
                    }
                    else
                    {
                        return Path.Combine(Environment.CurrentDirectory, Path.ChangeExtension(Path.GetFileName(InputFilePath), ".vcxproj"));
                    }
                }
                private set
                {
                    m_OutputFilePath = value;
                }
            }

            [CommandLineOption(
                "filters",
                Description = "プロジェクトフィルターファイルを出力します。",
                DefaultValue = false)]
            public bool CreateFilters
            {
                get
                {
                    return m_CreateFilters || CreateFiltersOnly;
                }
                private set
                {
                    m_CreateFilters = value;
                }
            }

            [CommandLineOption(
                "filters-only",
                Description = "フィルターファイルのみを出力します。プロジェクトファイルは生成しません。",
                DefaultValue = false)]
            public bool CreateFiltersOnly { get; private set; }

            [CommandLineOption(
                'f',
                Description = "出力ファイルが既に存在する場合に、上書き更新します。",
                DefaultValue = false)]
            public bool ForceOverwrite { get; private set; }

            [CommandLineOption(
                "path-property-file",
                Description = "パスプロパティ定義ファイルのパスです。")]
            public string PathPropertyFilePath { get; private set; }

            [CommandLineOption(
                "use-builtin-definitions",
                Description = "ツールチェインのコマンドラインオプション定義を実行時に生成せず、組み込みの定義を使用します。",
                DefaultValue = false)]
            public bool UseBuiltinOptionDefinitions { get; private set; }

            [CommandLineOption(
                "info",
                Description = "情報レベルのログ出力を有効にします。",
                DefaultValue = false)]
            public bool EnableInfoLogging { get; private set; }

            [CommandLineOption(
                "detail-embed-configuration-validity-flag",
                Description = "非公開機能です。有効でないプロジェクト構成に対して _NintendoSdkIsUnsupportedProjectConfiguration プロパティを定義します。",
                DefaultValue = false,
                IsHidden = true)]
            public bool EmbedInvalidConfigurationFlag { get; private set; }

            [CommandLineOption(
                "detail-embed-nnsdk-library-project-definition",
                Description = "非公開機能です。NintendoSDK のライブラリプロジェクト用の各種定義を追加します。",
                DefaultValue = false,
                IsHidden = true)]
            public bool EmbedNintendoSdkLibraryProjectDefinition { get; private set; }

            [CommandLineOption(
                "detail-embed-nnsdk-sample-project-definition",
                Description = "非公開機能です。NintendoSDK のサンプルプロジェクト用の各種定義を追加します。",
                DefaultValue = false,
                IsHidden = true)]
            public bool EmbedNintendoSdkSampleProjectDefinition { get; private set; }

            [CommandLineOption(
                "detail-embed-nnsdk-test-project-definition",
                Description = "非公開機能です。NintendoSDK の単体テストプロジェクト用の各種定義を追加します。",
                DefaultValue = false,
                IsHidden = true)]
            public bool EmbedNintendoSdkTestProjectDefinition { get; private set; }

            public bool CreateProject
            {
                get
                {
                    return !CreateFiltersOnly;
                }
            }

            public string FiltersFilePath
            {
                get
                {
                    return OutputFilePath + ".filters";
                }
            }

            public ICommand CreateCommand()
            {
                return new GenerateCommand()
                {
                    Parameter = this
                };
            }
        }

        private PlatformElementConverterComposer m_Composer = new PlatformElementConverterComposer();

        public CommandParameter Parameter { get; private set; }

        public void Execute()
        {
            Log.EnableInfoLogging = Parameter.EnableInfoLogging;

            var projectSetting = ReadProjectSetting();

            var pathProperties = ReadPathProperties();

            if (Parameter.CreateProject)
            {
                var projectRootElement = GenerateProjectRootElement(projectSetting, pathProperties);

                SaveProjectFile(projectRootElement, Parameter.OutputFilePath);
            }
            if (Parameter.CreateFilters)
            {
                var filtersRootElement = GenerateProjectFiltersRootElement(projectSetting);

                SaveProjectFile(filtersRootElement, Parameter.FiltersFilePath);
            }
        }

        private ProjectSetting ReadProjectSetting()
        {
            ProjectSetting projectSetting;
            try
            {
                projectSetting = ProjectSetting.Read(Parameter.InputFilePath);
            }
            catch (System.IO.IOException ex)
            {
                Log.Error("プロジェクト設定ファイルの読み込みに失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;
            }
            catch (YamlDotNet.Core.YamlException ex)
            {
                Log.Error("プロジェクト設定ファイルの読み込みに失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;
            }
            catch
            {
                Log.Error("プロジェクト設定ファイルの読み込みに失敗しました");
                throw;
            }

            // yaml の仕様上、改行コードは必ず LF で正規化される
            // 改行を含むテキストがそのまま vcxproj の中で使われると Visual Studio 上での GUI 表示に問題が出るため、ここで明示的に CRLF に変換する
            foreach (var targetSetting in projectSetting.TargetSettings)
            {
                if (!string.IsNullOrEmpty(targetSetting.PreBuildEvent))
                {
                    targetSetting.PreBuildEvent = NormalizeLineBreak(targetSetting.PreBuildEvent);
                }
                if (!string.IsNullOrEmpty(targetSetting.PreLinkEvent))
                {
                    targetSetting.PreLinkEvent = NormalizeLineBreak(targetSetting.PreLinkEvent);
                }
                if (!string.IsNullOrEmpty(targetSetting.PostBuildEvent))
                {
                    targetSetting.PostBuildEvent = NormalizeLineBreak(targetSetting.PostBuildEvent);
                }
            }

            return projectSetting;
        }

        private string NormalizeLineBreak(string text)
        {
            return Regex.Replace(text, @"\r\n|\r|\n", Environment.NewLine);
        }

        private IEnumerable<ProjectProperty> ReadPathProperties()
        {
            if (string.IsNullOrEmpty(Parameter.PathPropertyFilePath))
            {
                return Enumerable.Empty<ProjectProperty>();
            }

            try
            {
                using (var stream = new StreamReader(Parameter.PathPropertyFilePath))
                {
                    var deserializer = new Deserializer();
                    return deserializer.Deserialize<IEnumerable<ProjectProperty>>(stream);
                }
            }
            catch (System.IO.IOException ex)
            {
                Log.Error("パスプロパティ定義ファイルの読み込みに失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;
            }
            catch (YamlDotNet.Core.YamlException ex)
            {
                Log.Error("パスプロパティ定義ファイルの読み込みに失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;
            }
            catch
            {
                Log.Error("パスプロパティ定義ファイルの読み込みに失敗しました");
                throw;
            }
        }

        private ProjectRootElement GenerateProjectRootElement(ProjectSetting projectSetting, IEnumerable<ProjectProperty> pathProperties)
        {
            var projectBaseDirectory = Path.GetDirectoryName(Path.GetFullPath(Parameter.OutputFilePath));
            var substitutionPaths = PathUtilityExtension.GetSubstitutionPaths(pathProperties);

            var converters = m_Composer.EnumerateConverters();
            foreach (var converter in converters)
            {
                converter.SetProjectPathInfo(projectBaseDirectory, substitutionPaths);

                if (converter is IBuiltinOptionDefinitionsAvailable)
                {
                    ((IBuiltinOptionDefinitionsAvailable)converter).UseBuiltinOptionDefinitions = Parameter.UseBuiltinOptionDefinitions;
                }
            }

            try
            {
                var projectGenerator = new ProjectGenerator(projectSetting, converters);
                projectGenerator.EmbedConfigurationValidityFlag = Parameter.EmbedInvalidConfigurationFlag;
                projectGenerator.EmbedNintendoSdkLibraryProjectDefinition = Parameter.EmbedNintendoSdkLibraryProjectDefinition;
                projectGenerator.EmbedNintendoSdkSampleProjectDefinition = Parameter.EmbedNintendoSdkSampleProjectDefinition;
                projectGenerator.EmbedNintendoSdkTestProjectDefinition = Parameter.EmbedNintendoSdkTestProjectDefinition;
                return projectGenerator.Generate(projectBaseDirectory, substitutionPaths);
            }
            catch (ProjectGenerationException ex)
            {
                Log.Error("プロジェクトファイルの生成に失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;
            }
            catch
            {
                Log.Error("プロジェクトファイルの生成に失敗しました");
                throw;
            }
        }

        private ProjectRootElement GenerateProjectFiltersRootElement(ProjectSetting projectSetting)
        {
            var projectBaseDirectory = Path.GetDirectoryName(Path.GetFullPath(Parameter.OutputFilePath));
            var pathUtility = new PathUtility(projectBaseDirectory);

            try
            {
                return ProjectFiltersGenerator.Generate(projectSetting, pathUtility);
            }
            catch
            {
                Log.Error("プロジェクトフィルターファイルの生成に失敗しました");
                throw;
            }
        }

        private void SaveProjectFile(ProjectRootElement projectRootElement, string outputPath)
        {
            if (File.Exists(outputPath) && !Parameter.ForceOverwrite)
            {
                Log.Error(
                    "プロジェクトファイル '{0}' は既に存在します。上書きする場合は -f オプションを指定してください。",
                    Path.GetFileName(outputPath));
                Environment.Exit(1);
            }

            string tempFile;
            do
            {
                tempFile = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            }
            while (File.Exists(tempFile) || Directory.Exists(tempFile));

            try
            {
                projectRootElement.Save(tempFile);

                File.Delete(outputPath);
                File.Move(tempFile, outputPath);
            }
            catch (Exception ex)
            {
                Log.Error("プロジェクトファイルの書き出しに失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
            }
        }
    }
}
