﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    public abstract class OptionStringConverter : IOptionStringConverter
    {
        protected string SwitchPrefix { get; set; }
        protected IEnumerable<OptionDefinition> OptionDefinitions { get; set; }

        // TODO: 同じような実装のメソッドを共通化する

        public IEnumerable<OptionValue> ConvertToOptionValue(IEnumerable<string> ops, out IEnumerable<string> unknowns)
        {
            var target = new List<string>(ops);
            var unknown = new List<string>();
            var converted = new List<OptionValue>();

            while (target.Count > 0)
            {
                var op = target.First();

                IEnumerable<string> consumed;
                var optionValues = ConvertToOptionValue(op, target, out consumed);

                if (optionValues != null)
                {
                    converted.AddRange(optionValues);
                    foreach (var c in consumed.ToArray())
                    {
                        target.Remove(c);
                    }
                }
                else
                {
                    target.Remove(op);
                    unknown.Add(op);
                }
            }

            unknowns = unknown;
            return converted;
        }

        public IEnumerable<string> ConvertToOptionString(IEnumerable<OptionValue> ovs, out IEnumerable<OptionValue> unknowns)
        {
            var target = new List<OptionValue>(ovs);
            var unknown = new List<OptionValue>();
            var converted = new List<string>();

            while (target.Count > 0)
            {
                var ov = target.First();

                IEnumerable<OptionValue> consumed;
                var optionStrings = ConvertToOptionString(ov, target, out consumed);

                if (optionStrings != null)
                {
                    converted.AddRange(optionStrings);
                    foreach (var c in consumed.ToArray())
                    {
                        target.Remove(c);
                    }
                }
                else
                {
                    target.Remove(ov);
                    unknown.Add(ov);
                }
            }

            unknowns = unknown;
            return converted;
        }

        protected virtual IEnumerable<OptionValue> ConvertToOptionValue(string os, IEnumerable<string> all, out IEnumerable<string> consumed)
        {
            // オプション文字列にマッチするオプション定義は複数存在する可能性がある (*) ため、最もスイッチの長さが
            // 長いものを返すようにする
            //
            // (*) 例えば、引数を取るオプション /Switch と引数を取らないオプション /SwitchArg の定義がある時、
            //     入力 /SwitchArg は以下のように、いずれの定義にもマッチする
            //        * スイッチ /Switch と引数 Arg の組み合わせとして
            //        * スイッチ /SwitchArg として
            //     これらは本質的に区別できないため、とりあえずスイッチの長さが最も長いものを採用する
            //     (上の例の場合、引数を取らないオプション /SwitchArg が入力されたと見なす)

            var candidates = new List<Tuple<OptionDefinition, IEnumerable<OptionValue>, IEnumerable<string>>>();

            // オプション文字列からの変換では Switch が空のオプションは識別できないため無視する
            var ods = OptionDefinitions.Where(x => !string.IsNullOrEmpty(x.Switch));
            foreach (var od in ods)
            {
                IEnumerable<string> c;
                var optionValues = ConvertToOptionValue(os, all, od, out c);
                if (optionValues != null)
                {
                    candidates.Add(new Tuple<OptionDefinition, IEnumerable<OptionValue>, IEnumerable<string>>(od, optionValues, c));
                }
            }

            if (candidates.Count > 0)
            {
                var valueWithLongestSwitch = candidates.OrderByDescending(x => x.Item1.Switch.Length).First();
                consumed = valueWithLongestSwitch.Item3;
                return valueWithLongestSwitch.Item2;
            }
            else
            {
                consumed = new string[] { os };
                return null;
            }
        }

        protected virtual IEnumerable<OptionValue> ConvertToOptionValue(
            string os, IEnumerable<string> all, OptionDefinition od, out IEnumerable<string> consumed)
        {
            consumed = new string[] { os };

            // スイッチと引数を空白で分割する場合、合致するパターンは以下の二通りのはず
            //  * os は Tokenize されており、引数が取り除かれている ---> スイッチ名と完全一致する
            //  * os は Tokenize されておらず、空白を挟んで引数が末尾に付加されている ---> 空白を含むスイッチ名と先頭が一致する
            if (od.ShouldSeparateSwitchAndArguments
                && !(os == od.Switch || os.StartsWith(od.Switch + " ")))
            {
                return null;
            }

            // スイッチと引数を空白で分割しない場合、単にスイッチ名の先頭が一致するかどうかを見ればよいはず
            if (!od.ShouldSeparateSwitchAndArguments && !os.StartsWith(od.Switch))
            {
                return null;
            }

            if (!od.AcceptsArgument)
            {
                if (false
                    || (od.ShouldSeparateSwitchAndArguments && (os == od.Switch || os == od.Switch + " "))
                    || (!od.ShouldSeparateSwitchAndArguments && os == od.Switch))
                {
                    return new OptionValue[] { new OptionValue(od.Switch) };
                }
                else
                {
                    return null;
                }
            }

            var argument = GetOptionArgument(os, all, od, out consumed);

            if (od.IsArgumentRequired && argument == null)
            {
                consumed = new string[] { os };
                return null;
            }

            if (argument != null)
            {
                return new OptionValue[] { new OptionValue(od.Switch, Unquote(argument)) };
            }
            else
            {
                return new OptionValue[] { new OptionValue(od.Switch) };
            }
        }

        protected virtual IEnumerable<string> ConvertToOptionString(OptionValue ov, IEnumerable<OptionValue> all, out IEnumerable<OptionValue> consumed)
        {
            foreach (var od in OptionDefinitions)
            {
                var optionStrings = ConvertToOptionString(ov, all, od, out consumed);
                if (optionStrings != null)
                {
                    return optionStrings;
                }
            }

            consumed = new OptionValue[] { ov };
            return null;
        }

        protected virtual IEnumerable<string> ConvertToOptionString(
            OptionValue ov, IEnumerable<OptionValue> all, OptionDefinition od, out IEnumerable<OptionValue> consumed)
        {
            consumed = new OptionValue[] { ov };

            if (ov.Switch != od.Switch)
            {
                return null;
            }

            if (!od.AcceptsArgument && ov.HasArgument)
            {
                return null;
            }
            if (od.IsArgumentRequired && !ov.HasArgument)
            {
                return null;
            }

            if (ov.HasArgument)
            {
                // ArgumentName が定義されていれば該当する OptionDefinition を探す
                OptionDefinition ad = null;
                if (od.HasArgumentName)
                {
                    ad = OptionDefinitions.FirstOrDefault(x => x.MetadataName == od.ArgumentName);
                }

                return new string[] { string.Concat(
                    ov.Switch,
                    od.ShouldSeparateSwitchAndArguments ? " " : string.Empty,
                    od.Separator ?? string.Empty,
                    ad?.Separator ?? string.Empty,
                    Quote(ov.Argument)) };
            }
            else
            {
                return new string[] { ov.Switch };
            }
        }

        protected string GetOptionArgument(string os, IEnumerable<string> all, OptionDefinition od, out IEnumerable<string> consuming)
        {
            // ArgumentName が定義されていれば該当する OptionDefinition を探す
            OptionDefinition ad = null;
            if (od.HasArgumentName)
            {
                ad = OptionDefinitions.FirstOrDefault(x => x.MetadataName == od.ArgumentName);
            }

            // 文字列 os の中にスイッチと引数が一緒に含まれている場合は、引数のみを取り出す
            if (od.ShouldSeparateSwitchAndArguments)
            {
                var oss = OptionStringTokenizer.Tokenize(os);
                if (oss.Count() > 1)
                {
                    consuming = new string[] { os };
                    return string.Join(" ", oss.Skip(1));
                }
            }
            else
            {
                var switchLength = ad != null ? od.Switch.Length + ad.Separator.Length : od.Switch.Length;
                if (os.Length > switchLength)
                {
                    consuming = new string[] { os };
                    return os.Substring(switchLength);
                }
            }

            // os の中に引数が含まれていなければ、all の中から探し出す必要がある
            var index = all.ToList().IndexOf(os);
            if (index == -1)
            {
                throw new ArgumentException();
            }

            var argument = all.ElementAtOrDefault(index + 1);
            if (argument != null && !argument.StartsWith(SwitchPrefix))
            {
                consuming = new string[] { os, argument };
                return argument;
            }
            else
            {
                consuming = new string[] { os };
                return null;
            }
        }

        protected string Quote(string s)
        {
            return "\"" + Unquote(s) + "\"";
        }
        protected string Unquote(string s)
        {
            if (s.StartsWith("\"") && s.EndsWith("\"") && s.Length > 1)
            {
                return s.Substring(1, s.Length - 2);
            }
            else if (s.StartsWith("'") && s.EndsWith("'") && s.Length > 1)
            {
                return s.Substring(1, s.Length - 2);
            }
            else
            {
                return s;
            }
        }
    }
}
