﻿using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Construction;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    [Export(typeof(IPlatformElementConverter)), ExportMetadata("SupportedPlatform", "Cafe")]
    internal class CafePlatformElementConverter : PlatformElementConverter
    {
        // Converter を作ると同時にオプション定義を XML ファイルから読み込もうとするので、XML ファイルが存在しない
        // 環境ではエラーになる。必要があるまで Converter の生成は遅延させる
        private Lazy<CafeOptionConverter> m_CompilerOptionConverter = new Lazy<CafeOptionConverter>(() => CafeOptionConverter.GetCompilerOptionConverter());
        private Lazy<CafeOptionConverter> m_LibrarianOptionConverter = new Lazy<CafeOptionConverter>(() => CafeOptionConverter.GetLibrarianOptionConverter());
        private Lazy<CafeOptionConverter> m_LinkerOptionConverter = new Lazy<CafeOptionConverter>(() => CafeOptionConverter.GetLinkerOptionConverter());

        public override bool AcceptsBuildTargetSetting(BuildTargetSetting targetSetting)
        {
            return targetSetting.Platform == "Cafe";
        }

        protected override void AddCompileItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            base.AddCompileItemDefinition(itemDefinitionGroup, targetSetting);

            // プリコンパイル済みヘッダの設定だけデフォルト実装と微妙に違うため、設定値を上書きする
            if (HasValidPrecompiledHeaderSetting(targetSetting))
            {
                var clCompile = itemDefinitionGroup.ItemDefinitions.Single(x => x.ItemType == "ClCompile");

                // Win32 と仕様が違って、プロジェクトファイルからの相対パスでないと動作しない?
                clCompile.Metadata.Single(x => x.Name == "PrecompiledHeaderFile").Value = ConvertToRelativePath(targetSetting.PrecompiledHeaderFile);
            }
        }

        protected override void ReadPlatformSpecificItemDefinition(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            switch (itemDefinition.ItemType)
            {
                case "PreLink":
                    Log.WarnProjectElement(itemDefinition,
                        "プリリンカー設定の解釈には未対応であるため、無視します。");
                    break;
                case "PostLink":
                    Log.WarnProjectElement(itemDefinition,
                        "ポストリンカー設定の解釈には未対応であるため、無視します。");
                    break;
                default:
                    base.ReadPlatformSpecificItemDefinition(itemDefinition, targetSetting);
                    break;
            }
        }

        protected override OptionConverter GetCompileOptionConverter(BuildTargetSetting targetSetting)
        {
            return m_CompilerOptionConverter.Value;
        }

        protected override OptionConverter GetArchiveOptionConverter(BuildTargetSetting targetSetting)
        {
            return m_LibrarianOptionConverter.Value;
        }

        protected override OptionConverter GetLinkOptionConverter(BuildTargetSetting targetSetting)
        {
            return m_LinkerOptionConverter.Value;
        }

        protected override string CompilerName { get { return "MULTI コンパイラ"; } }

        protected override string ArchiverName { get { return "MULTI ライブラリアン"; } }

        protected override string LinkerName { get { return "MULTI リンカ"; } }

        protected override string LibraryFileExtension { get { return ".a"; } }
    }
}
