﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Construction;

namespace Nintendo.MakePropertySheets.Constructor
{
    internal class ProjectConstructor
    {
        public static ProjectRootElement Construct(string propsNameBody, IEnumerable<ProjectElementConstructor> elementConstructors, ReleaseIntent intent)
        {
            var props = ConstructWithProjectElementConstructors(elementConstructors, intent);

            InsertManualPropsImportElement(props, propsNameBody);

            return props;
        }

        private static ProjectRootElement ConstructWithProjectElementConstructors(
            IEnumerable<ProjectElementConstructor> elementConstructors, ReleaseIntent intent)
        {
            var props = new ProjectConstructionHelper();
            var ctx = new ProjectElementConstructor.ConstructionContext(intent);

            var elements = elementConstructors.SelectMany(x => x.Construct(ctx));

            var dependencyResolver = new ProjectElementConstructorDependencyResolver(elements);

            foreach (var elementGroup in dependencyResolver)
            {
                AddProjectElement(props, elementGroup);
                props.ForgetExistingGroups();
            }

            return props.Root;
        }

        private static void AddProjectElement(ProjectConstructionHelper props, IEnumerable<ProjectElement> elements)
        {
            foreach (var element in elements)
            {
                if (element is Property.ProjectProperty)
                {
                    var property = element as Property.ProjectProperty;

                    props.AddProperty(property.Name, property.Value, property.Condition);
                }
                else if (element is Property.ProjectPropertyGroup)
                {
                    var propertyGroup = element as Property.ProjectPropertyGroup;

                    var propsPropertyGroup = props.AddOrGetPropertyGroup(propertyGroup.Condition);

                    foreach (var property in propertyGroup.Properties)
                    {
                        propsPropertyGroup.AddProperty(property.Name, property.Value, property.Condition);
                    }
                }
                else if (element is ItemDefinition.ProjectMetadata)
                {
                    throw new ImplementationErrorException($"{nameof(ItemDefinition.ProjectMetadata)} is not used as an element of {nameof(ItemDefinition.ProjectItemDefinition)}");
                }
                else if (element is ItemDefinition.ProjectItemDefinition)
                {
                    var itemDefinition = element as ItemDefinition.ProjectItemDefinition;

                    var propsItemDefinition = props.AddOrGetItemDefinition(itemDefinition.ItemType, itemDefinition.Condition);

                    foreach (var metadata in itemDefinition.Metadatas)
                    {
                        propsItemDefinition.AddMetadata(metadata.Name, metadata.Value, metadata.Condition);
                    }
                }
                else if (element is Target.ProjectTarget)
                {
                    var target = element as Target.ProjectTarget;

                    var propsTarget = props.Root.AddTarget(target.Name);
                    propsTarget.Condition = target.Condition;
                    propsTarget.BeforeTargets = target.BeforeTargets;
                    propsTarget.AfterTargets = target.AfterTargets;

                    foreach (var task in target.Tasks)
                    {
                        var propsTask = propsTarget.AddTask(task.Name);
                        propsTask.Condition = task.Condition;

                        foreach (var kv in task.Parameters)
                        {
                            propsTask.SetParameter(kv.Key, kv.Value);
                        }
                    }
                }
                else if (element is Target.ProjectTask)
                {
                    throw new ImplementationErrorException($"{nameof(Target.ProjectTask)} is not used as an element of {nameof(Target.ProjectTarget)}");
                }
                else
                {
                    throw new NotImplementedException();
                }
            }
        }

        private static void InsertManualPropsImportElement(ProjectRootElement root, string propsNameBody)
        {
            var beforeProps = $"$(MSBuildThisFileDirectory){propsNameBody}.ImportBefore.props";
            root.PrependChild(root.CreateImportElement(beforeProps, $"exists('{beforeProps}')"));

            var afterProps = $"$(MSBuildThisFileDirectory){propsNameBody}.ImportAfter.props";
            root.AppendChild(root.CreateImportElement(afterProps, $"exists('{afterProps}')"));
        }

        private class ProjectConstructionHelper
        {
            private Dictionary<string, ProjectPropertyGroupElement> m_PropertyGroups;
            private Dictionary<string, ProjectItemDefinitionGroupElement> m_ItemDefinitionGroups;

            public ProjectRootElement Root { get; private set; }

            public ProjectConstructionHelper()
            {
                m_PropertyGroups = new Dictionary<string, ProjectPropertyGroupElement>();
                m_ItemDefinitionGroups = new Dictionary<string, ProjectItemDefinitionGroupElement>();

                Root = ProjectRootElement.Create();
            }

            public ProjectPropertyElement AddProperty(string name, string value, string condition = null)
            {
                var propertyGroup = AddOrGetPropertyGroup();
                return propertyGroup.AddProperty(name, value, condition);
            }

            public ProjectPropertyGroupElement AddOrGetPropertyGroup(string condition = null)
            {
                if (condition == null)
                {
                    condition = string.Empty;
                }

                var propertyGroup = default(ProjectPropertyGroupElement);
                if (!m_PropertyGroups.TryGetValue(condition, out propertyGroup))
                {
                    propertyGroup = Root.AddPropertyGroup(condition);
                    m_PropertyGroups.Add(condition, propertyGroup);
                }
                return propertyGroup;
            }

            public ProjectMetadataElement AddMetadata(string itemType, string name, string value, string condition = null)
            {
                var itemDefinition = AddOrGetItemDefinition(itemType);
                return itemDefinition.AddMetadata(name, value, condition);
            }

            public ProjectItemDefinitionElement AddOrGetItemDefinition(string itemType, string condition = null)
            {
                var itemDefinitionGroup = AddOrGetItemDefinitionGroup(condition);

                var itemDefinition = itemDefinitionGroup.ItemDefinitions.SingleOrDefault(x => x.ItemType == itemType);
                if (itemDefinition != null)
                {
                    return itemDefinition;
                }
                else
                {
                    return itemDefinitionGroup.AddItemDefinition(itemType);
                }
            }

            private ProjectItemDefinitionGroupElement AddOrGetItemDefinitionGroup(string condition = null)
            {
                if (condition == null)
                {
                    condition = string.Empty;
                }

                var itemDefinitionGroup = default(ProjectItemDefinitionGroupElement);
                if (!m_ItemDefinitionGroups.TryGetValue(condition, out itemDefinitionGroup))
                {
                    itemDefinitionGroup = Root.AddItemDefinitionGroup(condition);
                    m_ItemDefinitionGroups.Add(condition, itemDefinitionGroup);
                }
                return itemDefinitionGroup;
            }

            public void ForgetExistingGroups()
            {
                m_PropertyGroups.Clear();
                m_ItemDefinitionGroups.Clear();
            }
        }
    }
}
