﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include <iostream>
#include <sstream>
#include "Repository.h"
#include "CustomFilter.h"

namespace Libgit2 {

    class ExpandCheckoutProgressCallback
    {
    public:
        static void CallbackFunction(const char *path,
                                     size_t completed_steps,
                                     size_t total_steps,
                                     void *payload)
        {
            if(path)
            {
                // download スレッドの出力と混ざりにくくするため一度メモリ上に展開
                // 多少混ざっても致命的でないのでロックはとらない
                std::stringstream ss;
                ss<< "Expanding(" << completed_steps << "/" << total_steps << ")  " << path;
                std::cout << ss.str() << std::endl;
            }
        }
    };


    Repository::Repository(String^ workingDirectory)
        : m_WorkingDirectory(workingDirectory)
    {
        this->m_pRepository = new RepositoryHolder();
    }

    Repository::~Repository()
    {
        delete this->m_pRepository;
    }

    void Repository::Init()
    {
        String^ workingDirectory = this->m_WorkingDirectory;
        std::string repositoryPathStdString = marshal_as<std::string>(workingDirectory);

        int error = git_repository_init(this->m_pRepository->GetPointer(), repositoryPathStdString.c_str(), 0);

        if(error != 0)
        {
            throw gcnew Exception("fail to init repository.");
        }
    }

    void Repository::Open()
    {
        String^ workingDirectory = this->m_WorkingDirectory;
        std::string repositoryPathStdString = marshal_as<std::string>(workingDirectory);

        int error = git_repository_open_ext(this->m_pRepository->GetPointer(), repositoryPathStdString.c_str(), 0, NULL);

        if(error != 0)
        {
            throw gcnew Exception("fail to open repository.");
        }
    }

    String^ Repository::GetGitDirectory()
    {
        std::string path(git_repository_path(this->m_pRepository->GetHandle()));

        std::replace(path.begin(), path.end(), '\\', '/');

        return marshal_as<String^>(path);
    }

    String^ Repository::GetRootDirectory()
    {
        std::string workDirectoryPath(git_repository_workdir(this->m_pRepository->GetHandle()));

        std::replace(workDirectoryPath.begin(), workDirectoryPath.end(), '\\', '/');

        return marshal_as<String^>(workDirectoryPath);
    }

    Boolean Repository::IsInsideRepository()
    {
        String^ workingDirectory = this->m_WorkingDirectory;
        std::string workingDirectoryStdString = marshal_as<std::string>(workingDirectory);

        git_buf root = {0};
        int error = git_repository_discover(&root, workingDirectoryStdString.c_str(), 0, NULL);
        git_buf_free(&root);

        return error == 0;
    }

    String^ Repository::GetConfig(String^ name)
    {
        std::string nameStdString = marshal_as<std::string>(name);

        ConfigHolder config;
        if(git_repository_config(config.GetPointer(), this->m_pRepository->GetHandle()) != 0)
        {
            throw gcnew Exception("fail to get config manager.");
        }

        const char* configValue;
        if(git_config_get_string(&configValue, config.GetHandle(), nameStdString.c_str()) != 0)
        {
            throw gcnew Exception("fail to get config value.");
        }

        return marshal_as<String^>(configValue);
    }

    void Repository::SetConfig(String^ name, String^ value)
    {
        std::string nameStdString = marshal_as<std::string>(name);
        std::string valueStdString = marshal_as<std::string>(value);

        ConfigHolder config;
        if(git_repository_config(config.GetPointer(), this->m_pRepository->GetHandle()) != 0)
        {
            throw gcnew Exception("fail to get config manager.");
        }

        if(git_config_set_string(config.GetHandle(), nameStdString.c_str(), valueStdString.c_str()) != 0)
        {
            throw gcnew Exception("fail to set config value.");
        }
    }

    void Repository::Add(String^ filepath)
    {
        std::string filepathStdString = marshal_as<std::string>(filepath);

        IndexHolder index;
        if(git_repository_index(index.GetPointer(), this->m_pRepository->GetHandle()) != 0)
        {
            throw gcnew Exception("fail to get index.");
        }

        if(git_index_add_bypath(index.GetHandle(), filepathStdString.c_str()) != 0)
        {
            throw gcnew Exception("fail to add file.");
        }

        if(git_index_write(index.GetHandle()) != 0)
        {
            throw gcnew Exception("fail to write to index.");
        }
    }

    String^ Repository::ShowHead(String^ filepath)
    {
        std::string filepathStdString = marshal_as<std::string>(filepath);

        ObjectHolder object;
        std::string spec = "HEAD:" + filepathStdString;
        if(git_revparse_single(object.GetPointer(), this->m_pRepository->GetHandle(), spec.c_str()) != 0)
        {
            throw gcnew Exception("fail to rev-parse HEAD file.");
        }

        git_otype type = git_object_type(object.GetHandle());

        git_blob* blob = reinterpret_cast<git_blob*>(object.GetHandle());
        git_off_t rawsize = git_blob_rawsize(blob);
        const char *rawcontent = (const char*)git_blob_rawcontent(blob);

        msclr::interop::marshal_context context;
        String^ rawcontentDotNext = context.marshal_as<String^>(rawcontent);

        return rawcontentDotNext;
    }

    void Repository::Checkout(List<String^>^ filelist)
    {
        msclr::interop::marshal_context context;
        std::vector<char*> paths;

        for each (String^ file in filelist)
        {
            paths.push_back(const_cast<char*>(context.marshal_as<const char*>(file)));
        }

        git_checkout_options options = GIT_CHECKOUT_OPTIONS_INIT;
        options.checkout_strategy = GIT_CHECKOUT_FORCE;
        options.paths.strings = paths.data();
        options.paths.count = paths.size();

        if(git_checkout_head(this->m_pRepository->GetHandle(), &options) != 0)
        {
            throw gcnew Exception("fail to checkout.");
        }
    }

    void Repository::Expand(List<String^>^ filelist, String^ cacheDirectoryPath)
    {
        msclr::interop::marshal_context context;
        std::vector<char*> paths;

        if(filelist != nullptr)
        {
            for each (String^ file in filelist)
            {
                paths.push_back(const_cast<char*>(context.marshal_as<const char*>(file)));
            }
        }
        const char* cacheDirPathString = context.marshal_as<const char*>(cacheDirectoryPath);

        ExpandSmudgeFilter::Init(cacheDirPathString);
        ExpandSmudgeFilter::Register();
        try
        {
            git_checkout_options options = GIT_CHECKOUT_OPTIONS_INIT;
            options.checkout_strategy = GIT_CHECKOUT_FORCE;
            options.disable_filters = 0;

            if(paths.size() > 0)
            {
                options.paths.strings = paths.data();
                options.paths.count = paths.size();
            }

            options.progress_cb = ExpandCheckoutProgressCallback::CallbackFunction;
            options.progress_payload = NULL;

            int result = git_checkout_head(this->m_pRepository->GetHandle(), &options);
            if(result != 0)
            {
                throw gcnew Exception("fail to checkout.");
            }
        }
        catch(Exception^)
        {
            throw;
        }
        finally
        {
            ExpandSmudgeFilter::Shutdown();
        }
    }

    IEnumerable<TreeEntry^>^ Repository::GetHeadTree()
    {
        ObjectHolder object;
        if(git_revparse_single(object.GetPointer(), this->m_pRepository->GetHandle(), "HEAD^{tree}") != 0)
        {
            throw gcnew Exception("fail to rev-parse HEAD^{tree}.");
        }

        git_tree *tree = reinterpret_cast<git_tree*>(object.GetHandle());

        std::vector<TreeEntryNative> treeEntries;
        TreeWalker treeWalker(this->m_pRepository->GetHandle(), &treeEntries);

        if(git_tree_walk(tree, GIT_TREEWALK_PRE, TreeWalker::TreeWalkFunction, &treeWalker) != 0)
        {
            throw gcnew Exception("fail to walk tree.");
        }

        msclr::interop::marshal_context context;
        List<TreeEntry^>^ treeEntriesDotNet = gcnew List<TreeEntry^>();
        for(auto it = treeEntries.begin(); it != treeEntries.end(); it++)
        {
            TreeEntry^ entry = gcnew TreeEntry();
            entry->FilePath = context.marshal_as<String^>(it->filename);
            entry->FileSize = (int)it->size;
            treeEntriesDotNet->Add(entry);
        }

        return treeEntriesDotNet;
    }

    void Repository::SetAssumeUnchanged(List<String^>^ filelist, bool assumeUnchanged)
    {
        msclr::interop::marshal_context context;
        std::vector<char*> paths;

        if(filelist != nullptr)
        {
            for each (String^ file in filelist)
            {
                paths.push_back(const_cast<char*>(context.marshal_as<const char*>(file)));
            }
        }

        IndexHolder index;
        if(git_repository_index(index.GetPointer(), this->m_pRepository->GetHandle()) != 0)
        {
            throw gcnew Exception("fail to get index.");
        }

        for(auto path : paths)
        {
            const git_index_entry* indexEntry = git_index_get_bypath(index.GetHandle(), path, 0);
            if(indexEntry == nullptr)
            {
                throw gcnew Exception("fail to get index entry.");
            }

            git_index_entry indexEntryNew;
            memcpy(&indexEntryNew, indexEntry, sizeof(indexEntryNew));

            if(assumeUnchanged)
            {
                indexEntryNew.flags |= GIT_IDXENTRY_VALID;
            }
            else
            {
                indexEntryNew.flags &= ~GIT_IDXENTRY_VALID;
            }

            git_index_add(index.GetHandle(), &indexEntryNew);
        }

        if(git_index_write(index.GetHandle()) != 0)
        {
            throw gcnew Exception("fail to write to index.");
        }

        git_index_free(index.GetHandle());
    }
}
