﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using GitExternalStorage.Configs;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalStorage.Transports
{
    /// <summary>
    /// ファイルの転送方法を表現する基底クラスです。
    /// </summary>
    public abstract class TransportBase
    {
        private string cacheDirectoryPath;

        public TransportBase(string cacheDirectoryPath)
        {
            this.cacheDirectoryPath = cacheDirectoryPath;
        }

        /// <summary>
        /// ハッシュ値に対応するファイルをダウンロードします。
        /// </summary>
        /// <param name="sha">ファイルのハッシュ値</param>
        public void Pull(string sha)
        {
            var destFilePath = Path.Combine(this.cacheDirectoryPath, sha);

            GetFile(sha, destFilePath);
        }

        /// <summary>
        /// ハッシュ値に対応するファイルをダウンロードします。
        /// 一旦 tmpFilePath で指定したパスにダウンロードした後、本来のダウンロード場所に移動します。
        /// </summary>
        /// <param name="sha">ファイルのハッシュ値</param>
        /// <param name="tmpFilePath">一時ファイルを作成するパス。</param>
        public void PullUsingTempFile(string sha, string tmpFilePath)
        {
            var destFilePath = Path.Combine(this.cacheDirectoryPath, sha);
            try
            {
                GetFile(sha, tmpFilePath);
                File.Move(tmpFilePath, destFilePath);
            }
            finally
            {
                File.Delete(tmpFilePath);
            }
        }

        /// <summary>
        /// ハッシュ値に対応するファイルをアップロードします。
        /// </summary>
        /// <param name="sha">ファイルのハッシュ値</param>
        public void Push(string sha)
        {
            var sourceFilePath = Path.Combine(this.cacheDirectoryPath, sha);

            PutFile(sha, sourceFilePath);
        }

        /// <summary>
        /// 外部ストレージと同期の取られていないファイルを一覧します。
        /// </summary>
        /// <param name="files">検索対象のファイル群</param>
        /// <returns>外部ストレージ上に存在しないファイル群</returns>
        public abstract IEnumerable<string> GetUnpushed(IEnumerable<string> files);

        /// <summary>
        /// 外部ストレージ上にファイルが存在するかどうかを返します。
        /// </summary>
        /// <param name="sha">ファイルのハッシュ値</param>
        /// <returns>ファイルが存在する場合は True, しない場合は False</returns>
        public abstract bool Exists(string sha);

        /// <summary>
        /// 外部ストレージへのパスを返します。
        /// </summary>
        /// <returns>外部ストレージへのパス</returns>
        public abstract string GetPath();

        protected abstract void GetFile(string sha, string destFilePath);
        protected abstract void PutFile(string sha, string sourceFilePath);
    }
}
