﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalStorage.Git
{
    /// <summary>
    /// 外部 git コマンドを利用して機能を提供するクラスです。
    /// </summary>
    public class GitRepositoryCommand : GitRepositoryBase
    {
        private string workingDirectory;

        public GitRepositoryCommand(string workingDirectory)
        {
            this.workingDirectory = workingDirectory;
        }

        /// <summary>
        /// Git コマンドを実行します。
        /// </summary>
        /// <param name="options">コマンドに与える引数</param>
        /// <returns>コマンドの標準出力結果</returns>
        private string GetGitOutput(string options)
        {
            try
            {
                return ProcessUtility.GetProcessOutput("git", options, this.workingDirectory).Trim();
            }
            catch (ProcessUtilityExitCodeNonZeroException e)
            {
                throw new GitRepositoryCommandFailedException("git コマンドの実行に失敗しました。", e)
                {
                    Data = { { "Detail", string.Format("コマンド 'git {0}' の実行に失敗しました。", options) } }
                };
            }
            catch (ProcessUtilityWin32Exception e)
            {
                throw new GitRepositoryCommandNotFoundException("git コマンドが見つかりません。", e)
                {
                    Data = { { "Detail", string.Format("コマンド 'git {0}' が見つかりません。", options) } }
                };
            }
        }

        public override void Init()
        {
            this.GetGitOutput("init");
        }

        public override void Open()
        {
            // do nothing
        }

        public override string GetGitDirectory()
        {
            return GetGitOutput("rev-parse --git-dir");
        }

        public override string GetRepositoryRoot()
        {
            return GetGitOutput("rev-parse --show-toplevel");
        }

        public override bool IsInsideRepository()
        {
            try
            {
                return GetGitOutput("rev-parse --is-inside-work-tree") == "true" || GetGitOutput("rev-parse --is-inside-git-dir") == "true";
            }
            catch (GitRepositoryCommandFailedException)
            {
                // リポジトリ内にないとき、コマンドは "false" を出力するのではなく、
                // 終了ステータスとして 0 以外を吐いて失敗します。
                // そのため、ここで例外をハンドルし、"false" を返すようにします。
                return false;
            }
        }

        public override string GetConfig(string name)
        {
            return GetGitOutput(string.Format("config --get {0}", name));
        }

        public override void SetConfig(string name, string value)
        {
            GetGitOutput(string.Format("config {0} \"{1}\"", name, value));
        }

        public override IEnumerable<TreeEntry> GetHeadTree()
        {
            /*
             * ls-tree は、指定のリビジョンに含まれるファイルを列挙する
             *
             * -l, --long:
             *   ファイルサイズを表示する
             *
             * -z:
             *   各行をヌル文字で終端する
             *
             * -r:
             *   ツリーを再帰する
             *
             */
            var output = this.GetGitOutput("ls-tree -lz -r HEAD");

            // 行ごとにパースする。ls-tree の各行はヌル文字で終端されている。
            var outputLines = output.Split(new[] { '\0' }, StringSplitOptions.RemoveEmptyEntries);

            /*
             * 各行について、空白(SP)とタブ(TAB)でパースし、ReferenceEntry に格納する。
             * 以下、各行の出力フォーマット。
             *
             *   <mode> SP <type> SP <object> SP <object size> TAB <file>
             *
             */
            var delimiterCharsForLines = new[] { ' ', '\t' };
            var entries = outputLines.Select(x =>
            {
                var words = x.Split(delimiterCharsForLines, StringSplitOptions.RemoveEmptyEntries);
                if (words[1] == "blob")
                {
                    return new TreeEntry { FilePath = words[4], FileSize = int.Parse(words[3]) };
                }
                else
                {
                    return null;
                }
            }).Where(x => x != null);

            return entries;
        }

        public override void Add(string filepath)
        {
            this.GetGitOutput(string.Format("add \"{0}\"", filepath));
        }

        public override void Commit(string message)
        {
            this.GetGitOutput(string.Format("commit -m \"{0}\"", message));
        }

        public override string ShowHead(string filepath)
        {
            return this.GetGitOutput(string.Format("show HEAD:{0}", filepath));
        }

        public override void Checkout(string filepath)
        {
            this.GetGitOutput(string.Format("checkout HEAD -- \"{0}\"", filepath));
        }

        public override void Checkout(List<string> filelist)
        {
            var fileliststring = filelist.Select(x => '"' + x + '"').Aggregate((x, y) => x + " " + y);
            this.GetGitOutput(string.Format("checkout HEAD -- {0}", fileliststring));
        }

        public override void SetAssumeUnchanged(List<string> filelist, bool assumeUnchanged)
        {
            throw new NotImplementedException();
        }
    }
}
