﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using GitExternalStorage.Configs;
using GitExternalStorage.Transports;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using GitExternalStorage.Git;

namespace GitExternalStorage
{
    /// <summary>
    /// ツールの実行環境を表現します。
    /// </summary>
    public static class Environments
    {
        #region Environments Variables

        /// <summary>
        /// ツールディレクトリのパスを取得します。
        /// </summary>
        /// <returns>ツールディレクトリパス</returns>
        public static string GetExternalStorageDirectoryPath()
        {
            var path = Path.Combine(Environments.GitRepository.GetGitDirectory(), "GitExternalStorage");
            if (!Directory.Exists(path))
            {
                Directory.CreateDirectory(path);
            }

            return path;
        }

        /// <summary>
        /// キャッシュディレクトリのパスを取得します。
        /// </summary>
        /// <returns>キャッシュディレクトリパス</returns>
        public static string GetCacheDirectoryPath()
        {
            var path = Path.Combine(Environments.GitRepository.GetGitDirectory(), "GitExternalStorage", "Cache");
            if (!Directory.Exists(path))
            {
                Directory.CreateDirectory(path);
            }

            return path;
        }

        /// <summary>
        /// 一時ファイルの完全パスを返します。
        /// </summary>
        /// <returns>一時ファイルパス</returns>
        public static string GetTemporaryFilePath()
        {
            var temporaryDirectory = Path.Combine(Environments.GitRepository.GetGitDirectory(), "GitExternalStorage", "Temporary");
            if (!Directory.Exists(temporaryDirectory))
            {
                Directory.CreateDirectory(temporaryDirectory);
            }

            while (true)
            {
                var path = Path.Combine(temporaryDirectory, Path.GetRandomFileName());
                if (!File.Exists(path))
                {
                    return path;
                }
            }
        }

        #endregion

        #region GitRepository Singleton

        private static GitRepositoryCommand gitRepositoryCommand_;
        private static GitRepositoryLibgit2 gitRepositoryLibgit2_;

        /// <summary>
        /// <see cref="GitRepogitory"/> のインスタンスを返します。
        /// </summary>
        public static GitRepositoryBase GitRepository
        {
            get
            {
                return GitRepositoryLibgit2;
            }
        }

        public static GitRepositoryCommand GitRepositoryCommand
        {
            get
            {
                if (gitRepositoryCommand_ == null)
                {
                    gitRepositoryCommand_ = CreateGitRepositoryCommand();
                }

                return gitRepositoryCommand_;
            }
        }

        public static GitRepositoryLibgit2 GitRepositoryLibgit2
        {
            get
            {
                if (gitRepositoryLibgit2_ == null)
                {
                    gitRepositoryLibgit2_ = CreateGitRepositoryLibgit2();
                }

                return gitRepositoryLibgit2_;
            }
        }

        private static GitRepositoryCommand CreateGitRepositoryCommand()
        {
            try
            {
                var gitRepository = new GitRepositoryCommand(".");

                if (!gitRepository.IsInsideRepository())
                {
                    throw new GitExternalStorageException("The current directory is not inside git repository.");
                }

                gitRepository.Open();

                return gitRepository;
            }
            catch (GitRepositoryCommandNotFoundException e)
            {
                throw new GitExternalStorageException("git command does not exist.", e);
            }
        }

        private static GitRepositoryLibgit2 CreateGitRepositoryLibgit2()
        {
            try
            {
                var gitRepository = new GitRepositoryLibgit2(".");

                if (!gitRepository.IsInsideRepository())
                {
                    throw new GitExternalStorageException("The current directory is not inside git repository.");
                }

                gitRepository.Open();

                return gitRepository;
            }
            catch (GitRepositoryCommandNotFoundException e)
            {
                throw new GitExternalStorageException("git command does not exist.", e);
            }
        }

        #endregion

        #region Config Singleton

        private static ConfigStruct config_;

        /// <summary>
        /// <see cref="ConfigStruct"/> のインスタンスを返します。
        /// </summary>
        public static ConfigStruct Config
        {
            get
            {
                if (config_ == null)
                {
                    try
                    {
                        config_ = CreateConfigFromGit().ReadConfig();
                    }
                    catch (ConfigGitFormatException e)
                    {
                        throw new GitExternalStorageException(".git/config settings are wrong.", e);
                    }
                }

                return config_;
            }
        }

        public static ConfigFromGit CreateConfigFromGit()
        {
            return new ConfigFromGit(Environments.GitRepository);
        }

        #endregion

        #region Transport Singleton

        private static TransportBase transport_;

        /// <summary>
        /// <see cref="TransportBase"/> のインスタンスを返します。
        /// </summary>
        public static TransportBase Transport
        {
            get
            {
                if (transport_ == null)
                {
                    transport_ = CreateTransport();
                }

                return transport_;
            }
        }

        private static TransportBase CreateTransport()
        {
            switch (Environments.Config.Transport)
            {
                case TransportType.Local:
                    return new Transports.LocalTransport(Environments.GetCacheDirectoryPath(), Environments.Config.Local.Path);
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        #endregion

        #region CdnConfig Singleton

        private static bool cndConfigProcessed_;
        private static CdnConfigStruct cdnConfig_;

        /// <summary>
        /// <see cref="CdnConfigStruct"/> のインスタンスを返します。
        /// </summary>
        public static CdnConfigStruct CdnConfig
        {
            get
            {
                if (!cndConfigProcessed_)
                {
                    var config = CreateCdnConfigFromGit();

                    if (config.Exists())
                    {
                        try
                        {
                            cdnConfig_ = config.ReadConfig();
                        }
                        catch (ConfigGitFormatException e)
                        {
                            throw new GitExternalStorageException(".git/config settings are wrong.", e);
                        }
                    }

                    cndConfigProcessed_ = true;
                }

                return cdnConfig_;
            }
        }

        public static CdnConfigFromGit CreateCdnConfigFromGit()
        {
            return new CdnConfigFromGit(Environments.GitRepository);
        }

        #endregion

        #region Transport Singleton

        private static bool cdnTransportProcessed_;
        private static TransportBase cdnTransport_;

        /// <summary>
        /// CDN における <see cref="TransportBase"/> のインスタンスを返します。
        /// </summary>
        public static TransportBase CdnTransport
        {
            get
            {
                if (!cdnTransportProcessed_)
                {
                    cdnTransport_ = CreateCdnTransport();

                    cdnTransportProcessed_ = true;
                }

                return cdnTransport_;
            }
        }

        private static TransportBase CreateCdnTransport()
        {
            if (Environments.CdnConfig != null)
            {
                switch (Environments.CdnConfig.Transport)
                {
                    case TransportType.Local:
                        return new Transports.LocalTransport(Environments.GetCacheDirectoryPath(), Environments.CdnConfig.Local.Path);
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
            else
            {
                return null;
            }
        }

        #endregion
    }
}
