﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using GitExternalStorage.Consoles;
using GitExternalStorage.Core;
using GitExternalStorage.Transports;

namespace GitExternalStorage.Commands
{
    /// <summary>
    /// Git の filter-smudge に対応するコマンドです。
    /// </summary>
    public class FilterSmudgeCommand : CommandBase
    {
        public FilterSmudgeCommand()
            : base()
        {
        }

        public FilterSmudgeCommand(CommandConsoleBase console)
            : base(console)
        {
        }

        public override void Run()
        {
            using (var inputStream = CommandConsole.OpenInputStream())
            using (var reader = new BinaryReader(inputStream))
            using (var outputStream = CommandConsole.OpenOutputStream())
            {
                var contentBytes = reader.ReadBytes(HashedContent.GetContentSize());
                var contentString = Encoding.ASCII.GetString(contentBytes);
                if (HashedContent.IsMatch(contentString))
                {
                    var sha = HashedContent.Parse(contentString).Sha;
                    var cacheFilePath = Path.Combine(Environments.GetCacheDirectoryPath(), sha);

                    if (!File.Exists(cacheFilePath) && Environments.Config.AutoDownload)
                    {
                        CommandConsole.WriteLineToError("Downloading file: {0}", sha);

                        var downloaded = false;

                        // CDN 設定がある場合、まずはここからダウンロードすることを試す
                        if (Environments.CdnTransport != null)
                        {
                            var transport = Environments.CdnTransport;

                            CommandConsole.WriteLineToError("Try to download file: {0}/{1}", transport.GetPath(), sha);

                            try
                            {
                                transport.Pull(sha);

                                downloaded = true;
                            }
                            catch (Exception e)
                            {
                                CommandConsole.WriteLineToError("Info: failed to download file: {0}/{1}", transport.GetPath(), sha);
                                CommandConsole.WriteLineToError("Info: {0}", e.Message);
                            }
                        }

                        // CDN からダウンロードが行われていない、あるいは失敗した場合には、本体へアクセスします。
                        if (!downloaded)
                        {
                            var transport = Environments.Transport;

                            CommandConsole.WriteLineToError("Try to download file: {0}/{1}", transport.GetPath(), sha);

                            try
                            {
                                transport.Pull(sha);

                            }
                            catch (Exception)
                            {
                                CommandConsole.WriteLineToError("Error: failed to download file: {0}/{1}", transport.GetPath(), sha);

                                throw;
                            }
                        }
                    }

                    if (File.Exists(cacheFilePath))
                    {
                        CommandConsole.WriteLineToError("Recovering file: {0}", sha);
                        using (var fs = new FileStream(cacheFilePath, FileMode.Open))
                        {
                            fs.CopyTo(outputStream);
                        }
                    }
                    else
                    {
                        CommandConsole.WriteLineToError("File missing, saving placeholder: {0}", contentString);
                        outputStream.Write(contentBytes, 0, contentBytes.Count());
                        inputStream.CopyTo(outputStream);
                    }
                }
                else if (contentString == string.Empty)
                {
                    // ファイルサイズ 0 の場合、中身が空のファイルがコミットされているので、何も出力しません。
                    CommandConsole.WriteLineToError("This file size is 0.");
                }
                else
                {
                    CommandConsole.WriteLineToError("Unknown GitExternalStorage file format");
                    outputStream.Write(contentBytes, 0, contentBytes.Count());
                    inputStream.CopyTo(outputStream);
                }
            }
        }
    }
}
