﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepositoryManager.Exceptions.Utilities;

namespace GitExternalRepositoryManager.Utilities
{
    /// <summary>
    /// 外部プロセスを操作する機能を提供します。
    /// </summary>
    public class ProcessUtility
    {
        /// <summary>
        /// プロセスを起動して、その標準出力を取得します。
        /// </summary>
        /// <param name="command">起動するコマンド</param>
        /// <param name="options">コマンドに与える引数</param>
        /// <param name="workingDirectory">開始するコマンドの作業ディレクトリ</param>
        /// <returns>コマンドの標準出力結果</returns>
        public static string Run(string command, string options, string workingDirectory)
        {
            using (var process = new Process())
            {
                process.StartInfo = new ProcessStartInfo()
                {
                    FileName = command,
                    Arguments = options,
                    WorkingDirectory = workingDirectory,
                    UseShellExecute = false,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                };

                try
                {
                    process.Start();
                    var output = process.StandardOutput.ReadToEnd();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        var errorMessages = process.StandardError.ReadToEnd();

                        throw new ProcessUtilityExitCodeNonZeroException(string.Format("Failed to execute process: ExitCode = {0}. ErrorMessages = {1}", process.ExitCode, errorMessages))
                        {
                            Data = { { "Detail", string.Format("ExitCode = {0}. ErrorMessages = {1}", process.ExitCode, errorMessages) } }
                        };
                    }

                    return output;
                }
                catch (System.ComponentModel.Win32Exception e)
                {
                    throw new ProcessUtilityWin32Exception("Unknown error occurred. Failed to execute process.", e);
                }
            }
        }

        /// <summary>
        /// プロセスを起動して、その標準出力を取得します。
        /// 作業ディレクトリは、カレントディレクトリとなります。
        /// </summary>
        /// <param name="command">起動するコマンド</param>
        /// <param name="options">コマンドに与える引数</param>
        /// <returns>コマンドの標準出力結果</returns>
        public static string Run(string command, string options)
        {
            return Run(command, options, ".");
        }
    }
}
