﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace GitExternalRepository.Repository.Git
{
    /// <summary>
    /// git の config ファイルを git コマンドを用いて操作するクラスです。
    /// </summary>
    public class GitConfigCommand
    {
        public string FilePath { get; private set; }

        /// <summary>
        /// 指定の git config ファイルを操作するクラスを作成します。
        /// </summary>
        /// <param name="filepath">コンフィグファイルへのパス</param>
        public GitConfigCommand(string filepath)
        {
            this.FilePath = filepath;
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="name">名前</param>
        /// <param name="value">値</param>
        public void Set(string name, string value)
        {
            GitCommandUtils.GetGitOutput(".", string.Format("config  --file \"{0}\" {1} \"{2}\"", this.FilePath, name, value));
        }

        /// <summary>
        /// 値を取得します。
        /// </summary>
        /// <param name="name">名前</param>
        /// <returns>値</returns>
        public string Get(string name)
        {
            return GitCommandUtils.GetGitOutput(".", string.Format("config --file \"{0}\" --get {1}", this.FilePath, name));
        }

        /// <summary>
        /// セクションを削除します。
        /// </summary>
        /// <param name="name">名前</param>
        public void RemoveSection(string name)
        {
            GitCommandUtils.GetGitOutput(".", string.Format("config --file \"{0}\" --remove-section {1}", this.FilePath, name));
        }

        /// <summary>
        /// 全エントリを削除します。
        /// </summary>
        public void Clear()
        {
            File.WriteAllText(this.FilePath, string.Empty);
        }

        /// <summary>
        /// 正規表現を用いてエントリを探索します。この関数は、git config --get-regex コマンドの出力結果をそのまま返します。
        /// </summary>
        /// <param name="regex">正規表現</param>
        /// <returns>出力結果</returns>
        public string GetRegexRaw(string regex)
        {
            return GitCommandUtils.GetGitOutput(".", string.Format("config --file \"{0}\" --get-regex \"{1}\"", this.FilePath, regex));
        }

        /// <summary>
        /// 正規表現を用いてエントリを探索します。
        /// </summary>
        /// <param name="regex">正規表現</param>
        /// <returns>正規表現にマッチするエントリ</returns>
        public IEnumerable<GitConfigEntry> GetRegex(string regex)
        {
            var value = this.GetRegexRaw(regex);
            var regexWithSubsection = new Regex(@"^(.+)\.(.+)\.(.+) (.*)$");  // サブセクションあり
            var regexWithoutSubsection = new Regex(@"^(.+)\.(.+) (.*)$");     // サブセクションなし

            return this.ParseListupString(value, regexWithSubsection, regexWithoutSubsection);
        }

        /// <summary>
        /// すべての値をリストアップします。この関数は、git config --list コマンドの出力結果をそのまま返します。
        /// </summary>
        /// <returns>出力結果</returns>
        public string ListRaw()
        {
            return GitCommandUtils.GetGitOutput(".", string.Format("config --file \"{0}\" --list", this.FilePath));
        }

        /// <summary>
        /// すべての値をリストアップします。
        /// </summary>
        /// <returns>全エントリ</returns>
        public IEnumerable<GitConfigEntry> List()
        {
            var value = this.ListRaw();
            var regexWithSubsection = new Regex(@"^(.+)\.(.+)\.(.+)=(.*)$");  // サブセクションあり
            var regexWithoutSubsection = new Regex(@"^(.+)\.(.+)=(.*)$");     // サブセクションなし

            return this.ParseListupString(value, regexWithSubsection, regexWithoutSubsection);
        }

        /// <summary>
        /// すべてのセクションをリストアップします。
        /// </summary>
        /// <returns>全セクション</returns>
        public IEnumerable<GitConfigSection> ListSection()
        {
            var entries = this.List();

            var dictionary = new Dictionary<string, GitConfigSection>();

            foreach (var e in entries)
            {
                var key = e.Section + "." + e.SubSection;

                if (!dictionary.ContainsKey(key))
                {
                    dictionary.Add(key, new GitConfigSection()
                    {
                        Section = e.Section,
                        SubSection = e.SubSection,
                    });
                }

                if (dictionary[key].Entries.ContainsKey(e.Entry))
                {
                    // 同一のエントリが複数存在するときは、後ろのものが優先されるので、
                    // 以前のものは一旦削除します。
                    dictionary[key].Entries.Remove(e.Entry);
                }

                dictionary[key].Entries.Add(e.Entry, e.Value);
            }

            return dictionary.Values;
        }

        /// <summary>
        /// リストアップされたコンフィグ値をパースします。
        /// </summary>
        /// <param name="value">リストアップの結果</param>
        /// <param name="regexWithSubsection">サブセクションありの場合の正規表現</param>
        /// <param name="regexWithoutSubsection">サブセクションなしの場合の正規表現</param>
        /// <returns>パースした結果</returns>
        private IEnumerable<GitConfigEntry> ParseListupString(string value, Regex regexWithSubsection, Regex regexWithoutSubsection)
        {
            // 行にパース
            var lines = value.Split(new[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

            var entries = lines.Select(line =>
            {
                // サブセクションあり
                {
                    var match = regexWithSubsection.Match(line);
                    if (match.Success)
                    {
                        return new GitConfigEntry()
                        {
                            Section = match.Groups[1].Value,
                            SubSection = match.Groups[2].Value,
                            Entry = match.Groups[3].Value,
                            Value = match.Groups[4].Value,
                        };
                    }
                }

                // サブセクションなし
                {
                    var match = regexWithoutSubsection.Match(line);
                    if (match.Success)
                    {
                        return new GitConfigEntry()
                        {
                            Section = match.Groups[1].Value,
                            Entry = match.Groups[2].Value,
                            Value = match.Groups[3].Value,
                        };
                    }
                }

                throw new ArgumentException();
            });

            return entries;
        }
    }
}
