﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Commands;
using GitExternalRepository.Consoles;
using Nintendo.Foundation.IO;

namespace GitExternalRepository
{
    public class Program
    {
        private class Parameters
        {
            [CommandLineSubCommand("init", Description = "Initialize external repositories.")]
            public InitCommand.Parameters InitCommandParameters { get; set; }

            [CommandLineSubCommand("add", Description = "Add new external repository.")]
            public AddCommand.Parameters AddCommandParameters { get; set; }

            [CommandLineSubCommand("update", Description = "Update external repositories to the revision that is specified by .exremodules.")]
            public UpdateCommand.Parameters UpdateCommandParameters { get; set; }

            [CommandLineSubCommand("update-reference", Description = "Update the repository list with the current revision of the external repository.")]
            public UpdateReferenceCommand.Parameters UpdateReferenceCommandParameters { get; set; }

            [CommandLineSubCommand("clean", Description = "Clean external repositories. Delete all untracked or ignored files/directories.")]
            public CleanCommand.Parameters CleanCommandParameters { get; set; }

            [CommandLineSubCommand("list-tree", Description = "List tracked files in external repositories.")]
            public ListTreeCommand.Parameters ListTreeCommandParameters { get; set; }

            [CommandLineSubCommand("deinit", Description = "Deinitialize external repositories.")]
            public DeinitCommand.Parameters DeinitCommandParameters { get; set; }

            [CommandLineSubCommand("delete-unused-repository", Description = "Delete unused repositories.")]
            public DeleteUnusedRepositoryCommand.Parameters DeleteUnusedRepositoryCommandParameters { get; set; }

            public object GetValidCommandParameters()
            {
                return (object)this.InitCommandParameters
                    ?? (object)this.AddCommandParameters
                    ?? (object)this.UpdateCommandParameters
                    ?? (object)this.UpdateReferenceCommandParameters
                    ?? (object)this.CleanCommandParameters
                    ?? (object)this.ListTreeCommandParameters
                    ?? (object)this.DeinitCommandParameters
                    ?? (object)this.DeleteUnusedRepositoryCommandParameters;
            }
        }

        private static ConsoleBase CommandConsole { get; set; }

        public static void Main(string[] args)
        {
            // 入出力先となるコンソールを作成
            Program.CommandConsole = new StandardStreamConsole();

#if DEBUG
            Execute(args);
#else
            try
            {
                Execute(args);
            }
            catch (GitExternalRepositoryException e)
            {

                PrintExceptionDetail(e);

                Environment.Exit(1);
            }
            catch (CommandLineArgException)
            {
                // CommandLineParser がエラー出力するので、ここでは何も出力しない
                Environment.Exit(1);
            }
            catch (AggregateException e)
            {
                foreach (var exception in e.InnerExceptions)
                {
                    PrintExceptionDetail(exception);
                }

                Environment.Exit(1);
            }
            catch (Exception e)
            {
                PrintExceptionDetail(e);

                Environment.Exit(1);
            }
#endif
        }

        private static void PrintExceptionDetail(Exception exception)
        {
            var detailBuilder = new StringBuilder();
            for (var ex = exception; ex != null; ex = ex.InnerException)
            {
                if (ex.Data.Contains("Detail"))
                {
                    detailBuilder.Append(ex.Data["Detail"]);
                }
            }

            Program.CommandConsole.WriteLineToError("Error: {0}", exception.Message + detailBuilder.ToString());
            if (!(exception is GitExternalRepositoryException))
            {
                Program.CommandConsole.WriteLineToError(string.Empty);
                Program.CommandConsole.WriteLineToError("== Exception ==");
                Program.CommandConsole.WriteLineToError(string.Empty);
                Program.CommandConsole.WriteLineToError(exception.ToString());
            }
        }

        public static void Execute(string[] args)
        {
            var commandFactory = new CommandFactory();

            var settings = new CommandLineParserSettings()
            {
                ApplicationName = "git-exrepo",
                ApplicationDescription = "Manage external repositories.",
                HelpWriter = text => Program.CommandConsole.WriteToOut(text),
                ErrorWriter = text => Program.CommandConsole.WriteToError(text),
            };

            Parameters parameters;

            try
            {
                if (!new CommandLineParser(settings).ParseArgs<Parameters>(args, out parameters))
                {
                    return;
                }
            }
            catch (Exception exception)
            {
                throw new CommandLineArgException(exception);
            }

            // コマンドを生成し、実行します。
            var commandParameters = parameters.GetValidCommandParameters();
            if (commandFactory.Defines(commandParameters.GetType()))
            {
                var command = commandFactory.Create(commandParameters.GetType(), Program.CommandConsole);
                command.Execute(commandParameters);
            }
            else
            {
                throw new GitExternalRepositoryException("Failed to parse a command.");
            }
        }
    }
}
