﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using GitExternalRepository.Repository;
using GitExternalRepository.Repository.Git;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace GitExternalRepository.Exstorage
{
    public class ExstorageManager
    {
        /// <summary>
        /// 親から exstorage エイリアス設定をコピーし、コマンドをインストールします。
        ///
        /// 単にコピーするのではなく、外部リポジトリ上から正しくエイリアスコマンドを利用できるようパスの解消を行います。
        /// </summary>
        /// <param name="child"></param>
        /// <param name="parent"></param>
        /// <returns>エイリアスコマンドのインストールに成功した、あるいはすでにコマンドが存在した場合に true を返します。</returns>
        public bool InstallAliasFromParent(RepositoryBase child, RepositoryBase parent)
        {
            // 両方が git リポジトリである場合にのみ処理します。
            if (parent.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return false;
            }
            if (child.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return false;
            }

            try
            {
                var sourceFilterList = new ExstorageAliasListGitConfig(PathUtility.Combine(parent.GetMetaDirectory(), "config"));
                var sourceEntries = sourceFilterList.Read();
                var sourceEntry = sourceEntries.Count() > 0 ? sourceEntries.ElementAt(0) : null;

                var destFilterList = new ExstorageAliasListGitConfig(PathUtility.Combine(child.GetMetaDirectory(), "config"));
                var destEntries = destFilterList.Read();
                var destEntry = destEntries.Count() > 0 ? destEntries.ElementAt(0) : null;

                if (sourceEntry != null && sourceEntry.ExstorageAlias != null)
                {
                    var match = Regex.Match(sourceEntry.ExstorageAlias, @"!(.*)");
                    if (match.Success)
                    {
                        var command = match.Groups[1].Value;
                        var commandPath = Path.Combine(parent.GetRepositoryRoot(), command);
                        var commandPathFromDest = PathUtility.GetRelativePath(child.GetRepositoryRoot() + @"\", commandPath);

                        sourceEntry.ExstorageAlias = string.Format(@"!{0}", commandPathFromDest);
                    }

                    if (destEntry == null || !sourceEntry.Equals(destEntry))
                    {
                        // エイリアスをコピー
                        destFilterList.Add(sourceEntry);
                    }

                    return true;
                }

                return false;
            }
            catch (RepositoryOperationFailedException e)
            {
                throw new ExstorageManagerOperationFailedException("Failed to install exstorage-alias.", e);
            }
        }

        /// <summary>
        /// 親リポジトリにフィルタ設定がある場合に、子リポジトリにフィルタをインストールします。
        /// exstorage コマンドがインストールされている必要があります。
        /// </summary>
        /// <param name="repo"></param>
        /// <returns>インストールしたか、あるいはすでにインストール済みなら true を返します。</returns>
        public bool InstallFilterIfParentHas(RepositoryBase child, RepositoryBase parent)
        {
            // 両方が git リポジトリである場合にのみ処理します。
            if (parent.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return false;
            }
            if (child.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return false;
            }

            try
            {
                var sourceFilterList = new ExstorageFilterListGitConfig(PathUtility.Combine(parent.GetMetaDirectory(), "config"));
                var sourceEntries = sourceFilterList.Read();
                var sourceEntry = sourceEntries.Count() > 0 ? sourceEntries.ElementAt(0) : null;

                var destFilterList = new ExstorageFilterListGitConfig(PathUtility.Combine(child.GetMetaDirectory(), "config"));
                var destEntries = destFilterList.Read();
                var destEntry = destEntries.Count() > 0 ? destEntries.ElementAt(0) : null;

                if (sourceEntry != null)
                {
                    if (destEntry == null)
                    {
                        GitCommandUtils.GetGitOutput(child.GetRepositoryRoot(), "exstorage install-filter");
                    }

                    return true;
                }
                else
                {
                    return false;
                }
            }
            catch (RepositoryOperationFailedException e)
            {
                throw new ExstorageManagerOperationFailedException("Failed to install exstorage filter.", e);
            }
        }

        /// <summary>
        /// exstorage のフックスクリプトをインストールします。
        /// </summary>
        /// <param name="repo"></param>
        /// <returns></returns>
        public string InstallHookScript(RepositoryBase repo)
        {
            if (repo.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return null;
            }

            try
            {
                return GitCommandUtils.GetGitOutput(repo.GetRepositoryRoot(), "exstorage install-hooks");
            }
            catch (RepositoryOperationFailedException e)
            {
                throw new ExstorageManagerOperationFailedException("Failed to install exstorage hook script.", e);
            }
        }

        /// <summary>
        /// 指定のリポジトリ上で exstorage expand を実行します。
        /// </summary>
        /// <param name="repo"></param>
        /// <returns></returns>
        public string Expand(RepositoryBase repo)
        {
            if (repo.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return null;
            }

            try
            {
                return GitCommandUtils.GetGitOutput(repo.GetRepositoryRoot(), "exstorage expand");
            }
            catch (RepositoryOperationFailedException e)
            {
                throw new ExstorageManagerOperationFailedException("Failed to expand exstorage files.", e);
            }
        }

        /// <summary>
        /// GitExternalStorage 自身の設定をコピーします。
        ///
        /// また、単にコピーするのではなく、バイナリの配置場所の階層化を行います。
        /// 例えば、親リポジトリが \\localpath$\\hoge をバイナリの配置場所とするならば、 Externals/fuga に展開される外部リポジトリのバイナリの配置場所を \\localpath$\\hoge\\Externals\\fuga とします。
        /// </summary>
        /// <param name="relatedPathFromRoot"></param>
        /// <param name="child"></param>
        /// <param name="parent"></param>
        /// <returns>変更が発生したかどうか。以前と同じ値であった場合には false を返します。</returns>
        public bool CopyExstorageSettingsFromParent(string relatedPathFromRoot, RepositoryBase child, RepositoryBase parent)
        {
            // 両方が git リポジトリである場合にのみ処理します。
            if (parent.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return false;
            }
            if (child.GetRepositoryType() != Repository.RepositoryType.Git)
            {
                return false;
            }

            try
            {
                var modified = false;

                // GitExternalStorage のメイン設定をコピーします。
                {
                    var sourceFilterList = new ExstorageSettingsListGitConfig(PathUtility.Combine(parent.GetMetaDirectory(), "config"));
                    var sourceEntries = sourceFilterList.Read();
                    var sourceEntry = sourceEntries.Count() > 0 ? sourceEntries.ElementAt(0) : null;

                    var destFilterList = new ExstorageSettingsListGitConfig(PathUtility.Combine(child.GetMetaDirectory(), "config"));
                    var destEntries = destFilterList.Read();
                    var destEntry = destEntries.Count() > 0 ? destEntries.ElementAt(0) : null;

                    if (sourceEntry != null)
                    {
                        sourceEntry.LocalPath = Path.Combine(sourceEntry.LocalPath, relatedPathFromRoot);

                        if (destEntry == null || !sourceEntry.Equals(destEntry))
                        {
                            destFilterList.Add(sourceEntry);

                            modified |= true;
                        }
                    }
                }

                // GitExternalStorage の CDN 設定をコピーします。
                {
                    var sourceFilterList = new ExstorageSettingsCdnListGitConfig(PathUtility.Combine(parent.GetMetaDirectory(), "config"));
                    var sourceEntries = sourceFilterList.Read();
                    var sourceEntry = sourceEntries.Count() > 0 ? sourceEntries.ElementAt(0) : null;

                    var destFilterList = new ExstorageSettingsCdnListGitConfig(PathUtility.Combine(child.GetMetaDirectory(), "config"));
                    var destEntries = destFilterList.Read();
                    var destEntry = destEntries.Count() > 0 ? destEntries.ElementAt(0) : null;

                    if (sourceEntry != null)
                    {
                        sourceEntry.LocalPath = Path.Combine(sourceEntry.LocalPath, relatedPathFromRoot);

                        if (destEntry == null || !sourceEntry.Equals(destEntry))
                        {
                            destFilterList.Add(sourceEntry);

                            modified |= true;
                        }
                    }
                }

                return modified;
            }
            catch (RepositoryOperationFailedException e)
            {
                throw new ExstorageManagerOperationFailedException("Failed to copy 'GitExternalStorage' settings.", e);
            }
        }
    }
}
