﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Consoles;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// コマンドの生成を行うクラスです。
    /// </summary>
    public class CommandFactory
    {
        /// <summary>
        /// コマンドについての情報を表現するクラスです。
        /// </summary>
        private class CommandInformation
        {
            /// <summary>
            /// コマンドの型情報です。
            /// </summary>
            public Type Type { get; set; }

            /// <summary>
            /// コマンドの属性情報です。
            /// </summary>
            public CommandDefinitionAttribute Attirubte { get; set; }
        }

        private Dictionary<Type, CommandInformation> Commands { get; set; }

        public CommandFactory()
        {
            this.Commands = this.CreateCommandInformations();
        }

        /// <summary>
        /// コマンド情報を生成します。
        /// </summary>
        /// <returns>コマンド情報</returns>
        private Dictionary<Type, CommandInformation> CreateCommandInformations()
        {
            var commands = new Dictionary<Type, CommandInformation>();

            foreach (var type in Utils.GetInheritedClasses<CommandBase>())
            {
                var attr = (CommandDefinitionAttribute)Attribute.GetCustomAttribute(type, typeof(CommandDefinitionAttribute));

                if (attr == null)
                {
                    throw new CommandDefinitionAttributeNotDefinedException(string.Format("CommandBase を継承する {0} クラスは、属性 CommandDefinitionAttribute を定義しません。", type.Name));
                }

                var command = new CommandInformation()
                {
                    Type = type,
                    Attirubte = attr,
                };

                commands.Add(attr.ParameterType, command);
            }

            return commands;
        }

        /// <summary>
        /// 指定の名前のコマンドが定義されているかどうかを返します。
        /// </summary>
        /// <param name="parameterType">コマンドパラメータの型</param>
        /// <returns>定義されているなら True, そうでなければ False</returns>
        public bool Defines(Type parameterType)
        {
            return this.Commands.ContainsKey(parameterType);
        }

        /// <summary>
        /// 指定の名前を持つコマンドのインスタンスを生成します。
        /// </summary>
        /// <param name="parameterType">コマンドパラメータの型</param>
        /// <returns>コマンドのインスタンス</returns>
        public CommandBase Create(Type parameterType, ConsoleBase console)
        {
            if (!this.Defines(parameterType))
            {
                throw new ArgumentException(string.Format("サポートされていないコマンドパラメータの型が指定されました。", parameterType.Name));
            }

            return (CommandBase)System.Activator.CreateInstance(this.Commands[parameterType].Type, console);
        }
    }
}
