﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ResDic.h>
#include <nn/utilTool/utilTool_BinarizerContext2.h>

namespace nn { namespace utilTool {

BinarizerContext2::BinarizerContext2() NN_NOEXCEPT
    : m_Impl()
{
}

BinarizerContext2::~BinarizerContext2() NN_NOEXCEPT
{
    Reset();
}

void BinarizerContext2::Initialize( int sectionCount ) NN_NOEXCEPT
{
    m_Impl.Initialize( sectionCount );
    m_ResDicNodeList.reserve( 256 );
    m_PtrStringNodeList.reserve( 512 );
}

void BinarizerContext2::Reset() NN_NOEXCEPT
{
    m_Impl.Reset();

    for (ResDicNodeList::iterator iter = m_ResDicNodeList.begin(), end = m_ResDicNodeList.end(); iter != end; ++iter)
    {
        ResDicNode* pNode = *iter;
        pNode->keyList.clear();
        delete( pNode );
    }

    m_ResDicNodeList.clear();
    m_PtrStringNodeList.clear();
}

void* BinarizerContext2::GetPtr() NN_NOEXCEPT
{
    return m_Impl.GetBasePtr();
}

size_t BinarizerContext2::GetSize() const NN_NOEXCEPT
{
    return m_Impl.GetBaseSize() + m_Impl.GetRelocationTableSize();
}

size_t BinarizerContext2::GetAlignment() const NN_NOEXCEPT
{
    return m_Impl.GetAlignment();
}

void BinarizerContext2::SetName( const nn::util::string_view& name ) NN_NOEXCEPT
{
    m_Impl.SetName( name );
}

void BinarizerContext2::AddMemoryBlock( int sectionIndex, nn::util::MemorySplitter::MemoryBlock* pBlock ) NN_NOEXCEPT
{
    m_Impl.AddMemoryBlock( sectionIndex, pBlock );
}

void BinarizerContext2::AddResDicMemoryBlock(
    int sectionIndex,
    nn::util::MemorySplitter::MemoryBlock* pBlock,
    const nn::util::string_view* keys,
    int entryCount ) NN_NOEXCEPT
{
    m_Impl.AddMemoryBlock( sectionIndex, pBlock );

    ResDicNode* pNode = new ResDicNode();
    pNode->pBlock = pBlock;
    for (int i = 0; i < entryCount; i++)
    {
        pNode->keyList.push_back( m_Impl.InsertString( keys[i] ) );
    }
    m_ResDicNodeList.push_back(pNode);
}

nn::util::MemorySplitter::MemoryBlock* BinarizerContext2::CalculateStringPool() NN_NOEXCEPT
{
    nn::util::MemorySplitter::MemoryBlock* pStringPoolBlock = m_Impl.CalculateStringPool();

    // 文字列のオフセットが決まるので、文字列ポインタの登録を通常のポインタ登録に変換します。
    for ( PtrStringNodeList::iterator iter = m_PtrStringNodeList.begin(), end = m_PtrStringNodeList.end(); iter != end; ++iter )
    {
        m_Impl.RegisterBinPtr( iter->from, m_Impl.GetStringPosition( iter->str ) );
    }
    m_PtrStringNodeList.clear();

    return pStringPoolBlock;
}

void BinarizerContext2::SortByAlignment( int sectionIndex ) NN_NOEXCEPT
{
    m_Impl.SortByAlignment( sectionIndex );
}

void BinarizerContext2::RegisterBinPtrToString(
    const nn::util::MemorySplitter::MemoryBlock* pFrom,
    ptrdiff_t fromOffset,
    const nn::util::string_view& str ) NN_NOEXCEPT
{
    const nn::util::string_view inserted = m_Impl.InsertString(str);
    m_PtrStringNodeList.push_back( PtrStringNode( detail::MemberPosition( pFrom, fromOffset ), inserted ) );
}

void BinarizerContext2::RegisterBinPtr(
    const nn::util::MemorySplitter::MemoryBlock* pFrom,
    ptrdiff_t fromOffset,
    const nn::util::MemorySplitter::MemoryBlock* pTarget ) NN_NOEXCEPT
{
    RegisterBinPtr( pFrom, fromOffset, pTarget, 0 );
}

void BinarizerContext2::RegisterBinPtr(
    const nn::util::MemorySplitter::MemoryBlock* pFrom,
    ptrdiff_t fromOffset,
    const nn::util::MemorySplitter::MemoryBlock* pTarget,
    ptrdiff_t targetOffset ) NN_NOEXCEPT
{
    m_Impl.RegisterBinPtr( detail::MemberPosition( pFrom, fromOffset ), detail::MemberPosition( pTarget, targetOffset ) );
}

void BinarizerContext2::AddHeader( nn::util::BinaryBlockHeader* pHeader ) NN_NOEXCEPT
{
    m_Impl.AddHeader( pHeader );
}

nn::util::BinaryFileHeader* BinarizerContext2::Convert() NN_NOEXCEPT
{
    m_Impl.ConvertStringPool();
    m_Impl.ConvertBinPtr();
    ConvertResDic();
    m_Impl.ConvertRelocationTable();
    return m_Impl.ConvertHeaders();
}

void BinarizerContext2::Calculate() NN_NOEXCEPT
{
    const size_t EntriesOffset = offsetof( nn::util::ResDicData, nn::util::ResDicData::entries );
    const size_t KeyOffset = offsetof( nn::util::ResDicData::Entry, nn::util::ResDicData::Entry::pKey );
    const size_t EntrySize = sizeof(nn::util::ResDicData::Entry);

    m_Impl.CalculateBase();

    // リロケーションテーブルサイズの計算前に ResDic 内の BinPtrToString を登録します。
    for (ResDicNodeList::iterator iter = m_ResDicNodeList.begin(), end = m_ResDicNodeList.end(); iter != end; ++iter)
    {
        ResDicNode& node = **iter;

        for (size_t i = 0; i < node.keyList.size(); ++i)
        {
            // 先頭のエントリーは空文字用なので +1
            ptrdiff_t keyPtrOffset = static_cast< ptrdiff_t >( EntriesOffset + ( i + 1 ) * EntrySize + KeyOffset);
            m_Impl.RegisterBinPtr( detail::MemberPosition( node.pBlock, keyPtrOffset ), m_Impl.GetStringPosition( node.keyList[ i ] ) );
        }

        detail::MemberPosition rootPtrPos( node.pBlock, static_cast< ptrdiff_t >( EntriesOffset + KeyOffset ) );
        m_Impl.RegisterBinPtr( rootPtrPos, m_Impl.GetStringPosition( "" ) );
    }

    m_Impl.CalculateRelocationTable();
}

void BinarizerContext2::SetPtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT
{
    size_t baseSize = m_Impl.GetBaseSize();
    size_t relTableSize = m_Impl.GetRelocationTableSize();

    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, baseSize + relTableSize );
    NN_UNUSED( bufferSize );

    // その他のバッファ、サイズのアサートは Impl のチェックに任せる
    m_Impl.SetBasePtr( buffer, baseSize );
    m_Impl.SetRelocationTablePtr( nn::util::BytePtr( buffer, baseSize ).Get(), relTableSize );
}

void BinarizerContext2::ConvertResDic() NN_NOEXCEPT
{
    void* pBase = GetPtr();
    NN_SDK_ASSERT_NOT_NULL( pBase );

    for ( ResDicNodeList::iterator iter = m_ResDicNodeList.begin(), end = m_ResDicNodeList.end(); iter != end; ++iter)
    {
        ResDicNode& node = **iter;
        nn::util::ResDicData* pResDic = node.pBlock->Get< nn::util::ResDicData >( pBase );

        // pKey は ConvertBinPtr で既にオフセットを埋めているので、signature, count, Build だけでよい

        std::memcpy( pResDic->signature._str, "_DIC", 4 );
        pResDic->count = static_cast< int32_t >( node.keyList.size() );

        for (int i = 0; i < pResDic->count + 1; ++i)
        {
            pResDic->entries[ i ].pKey.Relocate( pBase ); // Build に必要なので Relocate
        }

        reinterpret_cast< nn::util::ResDic* >( pResDic )->Build();

        for ( int i = 0; i < pResDic->count + 1; ++i)
        {
            pResDic->entries[ i ].pKey.Unrelocate( pBase );
        }
    }
}

}} // namespace nn::utilTool
