﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <malloc.h>
#include <nn/utilTool/utilTool_BinarizerContext.h>

namespace nn { namespace utilTool {

BinarizerContext::BinarizerContext() NN_NOEXCEPT
    : m_Impl()
    , m_pAllocatedBase()
    , m_pAllocatedRelTable()
{
}

BinarizerContext::~BinarizerContext() NN_NOEXCEPT
{
    Reset();
}

void BinarizerContext::Initialize( int sectionCount ) NN_NOEXCEPT
{
    m_Impl.Initialize(sectionCount);
}

void BinarizerContext::Reset() NN_NOEXCEPT
{
    m_Impl.Reset();

    if ( m_pAllocatedBase )
    {
        _aligned_free( m_pAllocatedBase );
        m_pAllocatedBase = NULL;
    }

    if ( m_pAllocatedRelTable )
    {
        _aligned_free( m_pAllocatedRelTable );
        m_pAllocatedRelTable = NULL;
    }
}

void* BinarizerContext::GetBasePtr() NN_NOEXCEPT
{
    return m_Impl.GetBasePtr();
}

size_t BinarizerContext::GetBaseSize() const NN_NOEXCEPT
{
    return m_Impl.GetBaseSize();
}

size_t BinarizerContext::GetAlignment() const NN_NOEXCEPT
{
    return m_Impl.GetAlignment();
}

void* BinarizerContext::GetRelocationTablePtr() NN_NOEXCEPT
{
    return m_Impl.GetRelocationTablePtr();
}

size_t BinarizerContext::GetRelocationTableSize() const NN_NOEXCEPT
{
    return m_Impl.GetRelocationTableSize();
}

//--------------------------------------------------------------------------------------------------

void BinarizerContext::SetName( const nn::util::string_view& name ) NN_NOEXCEPT
{
    m_Impl.SetName( name );
}

void BinarizerContext::InsertString( const nn::util::string_view& str ) NN_NOEXCEPT
{
    m_Impl.InsertString( str );
}

void BinarizerContext::AddMemoryBlock( int sectionIndex, nn::util::MemorySplitter::MemoryBlock* pBlock ) NN_NOEXCEPT
{
    m_Impl.AddMemoryBlock( sectionIndex, pBlock );
}

nn::util::MemorySplitter::MemoryBlock* BinarizerContext::BuildStringPool() NN_NOEXCEPT
{
    return m_Impl.CalculateStringPool();
}

void BinarizerContext::SortByAlignment( int sectionIndex ) NN_NOEXCEPT
{
    m_Impl.SortByAlignment( sectionIndex );
}

//--------------------------------------------------------------------------------------------------

void BinarizerContext::ConvertStringPool() NN_NOEXCEPT
{
    m_Impl.ConvertStringPool();
}

void BinarizerContext::LinkString( nn::util::BinPtrToString* pBinPtr, const nn::util::string_view& str ) NN_NOEXCEPT
{
    nn::util::BinString* pString = m_Impl.GetBinString( str );
    LinkPtr( pBinPtr, pString );
}

void BinarizerContext::AddHeader( nn::util::BinaryBlockHeader* pHeader ) NN_NOEXCEPT
{
    m_Impl.AddHeader( pHeader );
}

//--------------------------------------------------------------------------------------------------

void BinarizerContext::ConvertRelocationTable() NN_NOEXCEPT
{
    m_Impl.ConvertRelocationTable();
}

nn::util::BinaryFileHeader* BinarizerContext::ConvertHeaders() NN_NOEXCEPT
{
    return m_Impl.ConvertHeaders();
}

void BinarizerContext::CalculateBase() NN_NOEXCEPT
{
    m_Impl.CalculateBase();
}

void BinarizerContext::AllocateBase() NN_NOEXCEPT
{
    CalculateBase();

    m_pAllocatedBase = _aligned_malloc( m_Impl.GetBaseSize(), m_Impl.GetAlignment() );
    this->SetBasePtr( m_pAllocatedBase, m_Impl.GetBaseSize() );
}

void BinarizerContext::SetBasePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT
{
    m_Impl.SetBasePtr( buffer, bufferSize );
}

void BinarizerContext::AllocateRelocationTable() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( m_Impl.GetBasePtr() );

    m_Impl.CalculateRelocationTable();

    const size_t RelTableAlignment = sizeof( nn::util::RelocationTable::AddrType );
    m_pAllocatedRelTable = _aligned_malloc( m_Impl.GetRelocationTableSize(), RelTableAlignment );
    this->SetRelocationTablePtr( m_pAllocatedRelTable, m_Impl.GetRelocationTableSize() );
}

void BinarizerContext::SetRelocationTablePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT
{
    m_Impl.SetRelocationTablePtr( buffer, bufferSize );
}

}} // namespace nn::utilTool
